// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "Components/BoxComponent.h"
#include "GameFramework/Actor.h"
#include "CuttingSectionScene.generated.h"


class UMaterialParameterCollection;

UENUM(BlueprintType)
enum ECuttingSectionID
{
	ESCI_Section1 UMETA(DisplayName="Cutting Section 1"),
	ESCI_Section2 UMETA(DisplayName="Cutting Section 2"),
	ESCI_Section3 UMETA(DisplayName="Cutting Section 3"),
	ESCI_Section4 UMETA(DisplayName="Cutting Section 4"),
	ESCI_Section5 UMETA(DisplayName="Cutting Section 5"),
	ESCI_Section6 UMETA(DisplayName="Cutting Section 6"),
	ESCI_Section7 UMETA(DisplayName="Cutting Section 7"),
};

UCLASS(/*ClassGroup=(IdealTwin), meta=(BlueprintSpawnableComponent), HideCategories=(Navigation,HLOD,Collision,Physics)*/)
class UCuttingSectionComponent : public UBoxComponent
{
	GENERATED_BODY()

public:
	UCuttingSectionComponent(const FObjectInitializer& ObjectInitializer);


	/**
	 * Controls whether to visualize the cutting plane in the editor viewport.
	 * For preview purposes only, has no effect at runtime.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="CuttingSectionVolume")
	bool bEditorUpdateCuttingVolume;
	/**
	 * Toggles the display of box outline for debugging purposes.
	 * Useful for visualizing the exact cutting boundaries in the editor.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="CuttingSectionVolume")
	bool bDebugBox;
	/**
	 * Defines the X, Y, Z dimensions of the cutting volume.
	 * Larger values create bigger cutting sections.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="CuttingSectionVolume")
	FVector ExtendCuttingBox;
	/**
	 * Determines if the cutting effect is active when the game begins.
	 * When true, objects will be clipped as soon as the level loads.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="CuttingSectionVolume")
	bool bStartCut;
	/**
	 * Identifies which cutting section this component represents.
	 * Each active cutting section must have a unique ID to prevent parameter conflicts.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="CuttingSectionVolume")
	TEnumAsByte<ECuttingSectionID> CuttingSectionId;

	/**
	 * Reference to the material parameter collection storing cutting plane data.
	 * Contains parameters for position, size, and rotation of all cutting sections.
	 */
	UPROPERTY(VisibleAnywhere, BlueprintReadOnly, Category="CuttingSectionVolume")
	TObjectPtr<UMaterialParameterCollection> CuttingSectionCollection;

private:
	bool bSetUpParameters;
};


/**
 * Creates a dynamic cutting section that reveals the interior of objects using material opacity.
 * This actor defines a volume that, when enabled, will create a cross-section view of any object
 * using compatible materials within the volume. Useful for architectural visualization, 
 * educational purposes, or gameplay mechanics requiring interior views.
 */
UCLASS()
class IDEALTWINPRO_API ACuttingSectionScene : public AActor
{
	GENERATED_BODY()

public:
	// Sets default values for this actor's properties
	ACuttingSectionScene(const FObjectInitializer& ObjectInitializer);

	
	UPROPERTY()
	TObjectPtr<UBoxComponent> BoxComponent;

	/**
	 * Controls whether to visualize the cutting plane in the editor viewport.
	 * For preview purposes only, has no effect at runtime.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="CuttingSectionVolume|Editor")
	bool bEditorUpdateCuttingVolume;
	/**
	 * Toggles the display of box outline for debugging purposes.
	 * Useful for visualizing the exact cutting boundaries in the editor.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="CuttingSectionVolume|Editor")
	bool bDebugBox;

	

	/**
	 * When enabled, the cutting volume will update its position every tick.
	 * Useful for cutting volumes that need to follow moving objects.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="CuttingSectionVolume")
	bool bTrackTickPosition;
	
	/**
	 * Defines the X, Y, Z dimensions of the cutting volume.
	 * Larger values create bigger cutting sections.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="CuttingSectionVolume")
	FVector ExtendCuttingBox;
	/**
	 * Determines if the cutting effect is active when the game begins.
	 * When true, objects will be clipped as soon as the level loads.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="CuttingSectionVolume")
	bool bStartCut;
	/**
	 * Identifies which cutting section this actor represents.
	 * Each active cutting section must have a unique ID to prevent parameter conflicts.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="CuttingSectionVolume")
	TEnumAsByte<ECuttingSectionID> CuttingSectionId;

	/**
	 * Reference to the material parameter collection storing cutting plane data.
	 * Contains parameters for position, size, and rotation of all cutting sections.
	 */
	UPROPERTY(VisibleAnywhere, BlueprintReadOnly, Category="CuttingSectionVolume")
	TObjectPtr<UMaterialParameterCollection> CuttingSectionCollection;

	/**
	 * Enables or disables all cutting section actors in the current level.
	 * Provides a convenient way to globally control all cutting sections.
	 * 
	 * @param WorldContext - Object providing access to the world
	 * @param bActive - True to enable all cutting sections, false to disable them
	 */
	UFUNCTION(BlueprintCallable, Category="CuttingSectionVolume", meta=(WorldContext="WorldContext"))
	static void SetLevelCuttingSectionActors(const UObject* WorldContext, bool bActive);

	/**
	 * Changes the ID of this cutting section.
	 * Useful for dynamically reassigning cutting sections at runtime.
	 * 
	 * @param InCuttingSectionId - The new ID to assign to this cutting section
	 */
	UFUNCTION(BlueprintCallable, Category="CuttingSectionVolume")
	void SetCuttingSectionId(TEnumAsByte<ECuttingSectionID> InCuttingSectionId);
	
	/**
	 * Updates the dimensions of the cutting box.
	 * 
	 * @param InExtend - New dimensions for the cutting box
	 * @param bUpdateAll - When true, updates all related parameters immediately
	 */
	UFUNCTION(BlueprintCallable, Category="CuttingSectionVolume")
	void SetCuttingExtends(FVector InExtend, bool bUpdateAll);

	/**
	 * Temporarily stops updating the cutting section position on tick.
	 * Useful for performance optimization or when precise control is needed.
	 */
	UFUNCTION(BlueprintCallable, Category="CuttingSectionVolume")
	void StopCuttingTrackPosition();
	
	/**
	 * Resumes updating the cutting section position on tick.
	 * Call after StopCuttingTrackPosition to restore automatic position updates.
	 */
	UFUNCTION(BlueprintCallable, Category="CuttingSectionVolume")
	void ResumeCuttingTrackPosition();
	
	/**
	 * Enables or disables the cutting effect for this section.
	 * 
	 * @param bActive - True to enable the cutting effect, false to disable it
	 * @param TrackPosition - When true, will continue updating position even when disabled
	 */
	UFUNCTION(BlueprintCallable, Category="CuttingSectionVolume")
	void SetActiveCuttingSection(bool bActive, bool TrackPosition = false);
	
	/**
	 * Manually updates the material parameters for this cutting section.
	 * Call this after moving or resizing the cutting section to apply changes.
	 */
	UFUNCTION(BlueprintCallable, Category="CuttingSectionVolume")
	void UpdateSectionCutVolume();

	/**
	 * Resets all material parameters associated with this cutting section.
	 * Effectively disables the cutting effect by clearing all parameter values.
	 */
	UFUNCTION(BlueprintCallable, Category="CuttingSectionVolume")
	void ResetParameters();

	//Actor interface
	virtual void Tick(float DeltaTime) override;
	virtual void Destroyed() override;
#if WITH_EDITOR
	virtual void PreEditChange(FProperty* PropertyAboutToChange) override;
	virtual void PostEditChangeProperty(FPropertyChangedEvent& PropertyChangedEvent) override;
	virtual void PostEditMove(bool bFinished) override;
#endif
protected:

	void GetCutVolumeIndexName(FString& CutPosition, FString& CutExtend, FString& CutRot) const;
	
	/**
	 * Initializes the cutting section based on the bStartCut property.
	 * Activates the cutting effect if bStartCut is true.
	 */
	virtual void BeginPlay() override;
private:

	bool bSetUpParameters;
};
