﻿// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "GameplayTagContainer.h"
#include "Slate/WidgetTransform.h"
#include "StructUtils/InstancedStruct.h"
#include "MinimapConfigAsset.generated.h"

class UMinimapIconWidget;


USTRUCT(BlueprintType) struct FMinimapWidgetSettings
{
	GENERATED_BODY()

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapWidgetSettings")
	FVector2D MinimapSize;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapWidgetSettings")
	FVector2D MinimapIconSize;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapWidgetSettings")
	bool bAutoAddPlayerIconToMinimap;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapWidgetSettings")
	bool bClampIconsToBorder;	
	
	FMinimapWidgetSettings();

	
	
};



USTRUCT(BlueprintType) struct FMinimapIconColor
{
	GENERATED_BODY()

	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="MinimapIconColor")
	FLinearColor NormalColor;
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="MinimapIconColor")
	FLinearColor HoveredColor;
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="MinimapIconColor")
	FLinearColor PressedColor;

	FMinimapIconColor();
};

USTRUCT(BlueprintType)
struct FMinimapIconSettings
{
	GENERATED_BODY()
	
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapIconSettings")
	UTexture2D* IconImage;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapIconSettings")
	FText DisplayName;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapIconSettings")
	FText IconDescription;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapIconSettings")
	bool bDynamic;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapIconSettings")
	FInstancedStruct MinimapIconColorsPayload;
	/** Icon Widget Class used to generate the icon into the minimap */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapIconSettings|Icon")
	TSubclassOf<UMinimapIconWidget> IconWidgetClassOverride;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapIconSettings|Icon")
	FVector2D IconSizeOverride;
	
	FMinimapIconSettings();
};

/**
 * Data structure for minimap floor data with texture and transform
 */
USTRUCT(BlueprintType)
struct FMinimapFloorData
{
	GENERATED_BODY()

	// Floor name displayed 
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "MinimapFloorData")
	FText FloorDisplayName = FText::FromString("Floor");
	
	// Texture to use for the floor representation
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "MinimapFloorData")
	UTexture2D* FloorTexture;

	// Widget transform for positioning and rotating the floor texture
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "MinimapFloorData")
	FWidgetTransform FloorTransform;

	FMinimapFloorData()
		: FloorTexture(nullptr)
		, FloorTransform(FWidgetTransform())
	{}
};

/**UMinimapConfigAsset is a data asset that holds configuration settings for the minimap system,
 * defining the behavior, appearance, and structure of minimap elements, including icons and floor data.*/
UCLASS()
class IDEALTWINPRO_API UMinimapConfigAsset : public UDataAsset
{
	GENERATED_BODY()
public:
	/**A map that associates gameplay tags with specific minimap icon settings.
	 * Each entry defines the appearance and behavior settings for minimap icons based on their associated tag.*/
	UPROPERTY(EditAnywhere, BlueprintReadWrite,  meta=(ForceInlineRow))
	TMap<FGameplayTag, FMinimapIconSettings> MinimapIconSettings;

	/**Array containing data for each floor in the minimap, including floor texture
	 * and its transform settings for proper display and alignment.*/
	UPROPERTY(EditAnywhere, BlueprintReadWrite, meta=(TitleProperty="{FloorDisplayName}"))
	TArray<FMinimapFloorData> MinimapFloors;

	UPROPERTY(EditAnywhere, BlueprintReadWrite)
	FMinimapWidgetSettings MinimapWidget;
	/** The default widget class used to represent an icon on the minimap.
	 * If no specific override is provided for an icon type in the associated minimap settings,
	 * this class is used to generate the icon widget.*/
	UPROPERTY(EditAnywhere, BlueprintReadWrite)
	TSubclassOf<UMinimapIconWidget> DefaultIconWidgetClass;
	
	
	UMinimapConfigAsset(const FObjectInitializer& ObjectInitializer);
};

