// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "GameFramework/Actor.h"
#include "MinimapFloorAnchor.generated.h"

/**
 * AMinimapFloorAnchor represents an anchor point used for determining the
 * position or alignment of the floor within a minimap system. It serves as
 * the reference location for capturing or aligning minimap visuals
 * relative to the environment's geometry.
 */
UCLASS(HideCategories=(HLOD,Physics,Collision,Networking,Input,Cooking))
class IDEALTWINPRO_API AMinimapFloorAnchor : public AActor
{
	GENERATED_BODY()

public:
	AMinimapFloorAnchor(const FObjectInitializer& ObjectInitializer);

	/**
	 * HeightOffset determines the vertical position of the minimap capture component
	 * relative to the actor's origin. Adjusting this value changes the height at
	 * which the floor is captured for the minimap visualization.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapFloorAnchor")
	float HeightOffset;
	/**
	 * CaptureArea defines the width and height of the orthographic view used by the minimap
	 * capture component. Increasing this value expands the area captured for the minimap,
	 * while decreasing it limits the viewable region.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapFloorAnchor")
	float CaptureArea;

	/** bDebugBounds is a flag that enables the visualization of the minimap capture area's boundaries
	 * within the editor's viewport. When set to true, the bounds of the capture area are drawn, aiding
	 * in debugging and adjustments to capture settings.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapFloorAnchor|Debug")
	bool bDebugBounds;

	/** FloorIndex represents the position of this anchor in the building's floor hierarchy.
	 * Lower values represent lower floors, higher values represent higher floors.
	 * A value of -1 indicates that the floor hasn't been assigned an index yet.
	 */
	UPROPERTY(VisibleAnywhere, BlueprintReadOnly, Category="MinimapFloorAnchor")
	int32 FloorIndex;
	
	virtual void Tick(float DeltaTime) override;
	virtual void Destroyed() override;
	
	/** Gets the floor capture component
	 * @return The floor capture component
	 */
	UFUNCTION(BlueprintCallable, Category="MinimapFloorAnchor")
	USceneCaptureComponent2D* GetFloorCaptureComponent() const { return FloorCaptureComponent; }
	
	/** Sets the texture target for the floor capture component
	 * @param NewTextureTarget - The new texture target
	 */
	UFUNCTION(BlueprintCallable, Category="MinimapFloorAnchor")
	void SetTextureTarget(UTextureRenderTarget2D* NewTextureTarget);
	
	/** Captures the current scene
	 */
	UFUNCTION(BlueprintCallable, Category="MinimapFloorAnchor")
	void CaptureScene();
	
	/** Gets the floor index of this floor anchor
	 * @return The floor index, or -1 if not assigned
	 */
	UFUNCTION(BlueprintCallable, Category="MinimapFloorAnchor")
	int32 GetFloorIndex() const { return FloorIndex; }
	
	/** Sets the floor index of this floor anchor
	 * @param InFloorIndex - The new floor index to assign
	 */
	UFUNCTION(BlueprintCallable, Category="MinimapFloorAnchor")
	void SetFloorIndex(int32 InFloorIndex) { FloorIndex = InFloorIndex; }

	UFUNCTION(BlueprintCallable, Category="MinimapFloorAnchor")
	void SetCaptureArea(float InCaptureArea);

	// When true, applies a post-process override to prevent render-target overexposure
	// caused by very high / physical light intensities (real-world light values).
	// This forces LocalExposureHighlightContrastScale to 0 to avoid blown-out capture results.
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="MinimapFloorAnchor")
	bool bFixedForRealLights;

#if WITH_EDITOR
	virtual void PostEditChangeProperty(struct FPropertyChangedEvent& PropertyChangedEvent) override;
	virtual void PostEditMove(bool bFinished) override;
	void DrawCaptureZone(bool bDraw) const;
	void SetFixForRealLights(bool bFix);
	/**
	 * Moves the actor by the specified delta and updates debug visualization if enabled
	 * @param NewLocation - The delta to apply to the actor's current location
	 */
	UFUNCTION(BlueprintCallable, Category="MinimapFloorAnchor|Editor")
	void AddDeltaToMove(FVector NewLocation);
#endif

protected:
	virtual void BeginPlay() override;
	
	UPROPERTY(VIsibleAnywhere, BlueprintReadOnly, Category="MinimapFloorAnchor")
	TObjectPtr<USceneCaptureComponent2D> FloorCaptureComponent;
};
