﻿#pragma once

#include "CoreMinimal.h"
#include "GameFramework/Actor.h"
#include "MinimapLevelController.generated.h"

class AMinimapFloorAnchor;
class AITwISPTeleportPoint;
class UMinimapConfigAsset;

DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnMinimapLevelChanged, int32, NewLevelIndex);

UCLASS(HideCategories=(HLOD,Physics,Collision,Networking,Input,Cooking))
class IDEALTWINPRO_API AMinimapLevelController : public AActor
{
	GENERATED_BODY()

public:
	
	AMinimapLevelController(const FObjectInitializer& ObjectInitializer);

	UPROPERTY(BlueprintAssignable, Category="MinimapLevelController|Events")
	FOnMinimapLevelChanged OnMinimapLevelChanged;

	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="MinimapLevelController")
	TSoftObjectPtr<UMinimapConfigAsset> MinimapConfiguration;
	
	UPROPERTY(EditAnywhere,BlueprintReadOnly ,Category="MinimapLevelController")
	TObjectPtr<UTextureRenderTarget2D> RenderTarget2D;
	
	UPROPERTY(EditInstanceOnly, Category="MinimapLevelController")
	FVector2D TargetResolution;

	/**When true the controller will manage ISP Teleport Point widgets at runtime/editor:
	 * - Calls RefreshAllISPTPPoints() during BeginPlay to gather and register teleport points.
	 * - Enables actor ticking so teleport points can be updated each tick.*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="MinimapLevelController")
	bool bManageISPTeleportPointsWidget;

	UFUNCTION(BlueprintCallable, Category="MinimapLevelController")
	void CaptureMinimapScene();
	
	UFUNCTION(BlueprintCallable, Category="MinimapLevelController")
	void SetCaptureArea(const float NewCaptureArea);
	
	UFUNCTION(BlueprintCallable, Category="MinimapLevelController")
	float GetCaptureArea() const;

	UFUNCTION(BlueprintCallable, Category="MinimapLevelController")
	UMinimapConfigAsset* GetMinimapConfiguration()const;

	
	/** Find all `MinimapFloorAnchor` actors in the current world. Clears and repopulates
	 * the `FloorAnchors` array, registers `OnDestroyed` handlers for each anchor, sorts
	 * them by height (Z coordinate), and updates `ActiveFloorIndex` (0 if anchors found,
	 * -1 if none). Use this to refresh the list when anchors are added, removed, or moved.
	 */
	UFUNCTION(CallInEditor, BlueprintCallable, Category="MinimapLevelController|Accions")
	void FindAllFloorAnchors();
	
	UTexture* GetFloorDisplayTexture();
	
	UFUNCTION(BlueprintCallable, Category="Minimap Floors")
	bool SetActiveFloor(int32 FloorIndex);
	
	UFUNCTION(BlueprintCallable, Category="Minimap Floors")
	int32 GetActiveFloorIndex() const;
	
	UFUNCTION(BlueprintCallable, Category="Minimap Floors")
	AMinimapFloorAnchor* GetActiveFloorAnchor() const;
	
	UFUNCTION(BlueprintCallable, Category="Minimap Floors")
	const TArray<AMinimapFloorAnchor*>& GetAllFloorAnchors() const;
	
	UFUNCTION(BlueprintCallable, Category="MinimapLevelController")
	UTextureRenderTarget2D* GetRenderTarget() const { return RenderTarget2D; }
	
	UFUNCTION(BlueprintCallable, Category="MinimapLevelController")
	void StartAutoFloorSelection(float UpdateInterval = 0.5f);
	UFUNCTION(BlueprintCallable, Category="MinimapLevelController") 
	void StopAutoFloorSelection();
	
	/*When true, instructs the minimap system to apply a fix for scenes using physical/real-world
	light intensities that can cause render-target captures to be overexposed. When a new floor
	anchor is created or discovered (via CreateNewFloorAnchor() or FindAllFloorAnchors()), this
	flag is copied to the anchor and SetFixForRealLights(bFixedForRealLights) is invoked on it.
	The anchor then overrides its SceneCapture post-process setting (forces
	LocalExposureHighlightContrastScale = 0) to prevent blown-out minimap render textures.*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="MinimapLevelController|Accions")
	bool bFixedForRealLights;
	// When true, newly created floor anchors will be automatically attached to this controller.
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="MinimapLevelController|Accions")
	bool bAutoAttachmentToManager;
	
	
	/**
     * Remove any registered handlers and clear the `FloorAnchors` array. This is an
     * editor helper to reset the controller's floor anchor list; it does not destroy actors.
     */
    UFUNCTION(CallInEditor, BlueprintCallable, Category="MinimapLevelController|Accions",
    	meta=(DisplayName="Clear Floor Anchor List"))
    void ClearFloorAnchorList();
    
    /**
     * Sorts `FloorAnchors` by their Z (height). After sorting, updates each anchor's index
     * and label (`Minimap_Floor_{i}`), adjusts `ActiveFloorIndex` if necessary, and reapplies
     * the active floor selection. Editor-only convenience to keep floors ordered.
     */
    UFUNCTION(CallInEditor, BlueprintCallable, Category="MinimapLevelController|Accions", meta=(DisplayName="Sort Floors By Height",DevelopmentOnly))
    void SortFloorAnchorsByHeight();
#if WITH_EDITOR
	virtual void PostEditChangeChainProperty(struct FPropertyChangedChainEvent& PropertyChangedEvent) override;
	virtual void PostEditMove(bool bFinished) override;
	UFUNCTION(BlueprintCallable, Category="MinimapLevelController|Editor")
	void AddDeltaToMove(FVector NewLocation);


	
	/**
	 * Creates a new `MinimapFloorAnchor` actor at this controller's location (editor only).
	 * Assigns the next available floor index and a descriptive label, optionally attaches
	 * the anchor to this controller if `bAutoAttachmentToManager` is true, updates the
	 * anchor list, and selects the new actor in the editor. No runtime behavior.
	 */
	UFUNCTION(CallInEditor, BlueprintCallable, Category="MinimapLevelController|Accions",
		meta=(DisplayName="Create New Floor Anchor"))
	void CreateNewFloorAnchor();

	

	// TODO: This panel is not working for a release version yet
	//UFUNCTION(CallInEditor, BlueprintCallable, Category="MinimapLevelController", meta=(DisplayName="Open Minimap Tool",DevelopmentOnly))
	void OpenMinimapTool();
	UFUNCTION(BlueprintCallable, Category="MinimapLevelController",meta=(DevelopmentOnly))
	void DebugFloorAnchor(bool bDebugBounds);

#endif

	virtual void Tick(float DeltaTime) override;
	virtual void Destroyed() override;
	
protected:
	virtual void BeginPlay() override;

	UFUNCTION() virtual void CheckPawnPositionAndSelectNearestFloor();
	UFUNCTION() void OnFloorAnchorDestroyed(AActor* DestroyedActor);

	void RefreshAllISPTPPoints();

	void BindToLevelVariantUpdate();

	/** In order to add more floors anchors use CreateNewFloorAnchor button or if you create then by placement use
		FindAllFloorAnchors to refresh the list */
	UPROPERTY(VisibleAnywhere, BlueprintReadOnly, Category="Minimap Floors")
	TArray<AMinimapFloorAnchor*> FloorAnchors;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="Minimap Floors")
	int32 ActiveFloorIndex;
	UPROPERTY(VisibleAnywhere, BlueprintReadOnly, Category="Minimap Floors")
	TObjectPtr<AMinimapFloorAnchor> ActiveFloorAnchor;

	UPROPERTY()
	TArray<AITwISPTeleportPoint*> ISPTeleportPoints;
private:
	UPROPERTY()
	TObjectPtr<UTexture2D> MinimapTextureTarget;
	
	FTimerHandle AutoFloorSelectionTimerHandle;
};
