// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "MinimapConfigAsset.h"
#include "Blueprint/UserWidget.h"

#include "MinimapWidget.generated.h"

class UMinimapConfigAsset;


UCLASS()
class IDEALTWINPRO_API UMinimapWidget : public UUserWidget
{
	GENERATED_BODY()

public:

	UMinimapWidget(const FObjectInitializer& ObjectInitializer);

	/*SC in charge of obtain the minimap image that will be displayed. It is initialized in InitMinimapRenderComponents method*/
	UPROPERTY(BlueprintReadOnly, Category="Minimap Widget")
	TObjectPtr<class AMinimapLevelController> MinimapController;

	/*Canvas panel in which the icons will be nested*/
	UPROPERTY(BlueprintReadOnly, meta=(BindWidget), Category="Minimap Widget")
	TObjectPtr<class UCanvasPanel> MinimapCanvas;

	/*Image in which the render target captured by the MinimapLevelController will be displayed*/
	UPROPERTY(BlueprintReadOnly, meta=(BindWidget), Category="Minimap Widget")
	TObjectPtr<class UImage> MinimapBackground;

	/*Allow the minimap to be resized in blueprint*/
	UPROPERTY(BlueprintReadOnly, meta=(BindWidget), Category="Minimap Widget")
	TObjectPtr<class USizeBox> MinimapSizeBox;

	/*Material used to render the render target. It needs to has a parameter called MinimapTexture*/
	UPROPERTY(EditAnywhere, Category="Minimap Widget")
	TObjectPtr<UMaterial> MinimapMaterial;

	


	/*Vector2D exposed to set the minimap size in blueprint*/
	UPROPERTY(EditAnywhere, Category="Minimap Widget")
	FVector2D MinimapSize;

	/*Vector2D exposed to set the minimap icon size in blueprint*/
	UPROPERTY(EditAnywhere, Category = "Minimap Widget")
	FVector2D MinimapIconSize;

	/*If true the minimap will automatically call InitializeMinimap, if not, call it from blueprint will be needed */
	UPROPERTY(EditAnywhere, Category="Minimap Widget|Autoinitialization")
	bool bAutoInitialized;

	/*How many seconds will pass until the minimap is initialized when bAutoInitialized is true*/
	UPROPERTY(EditAnywhere, Category="Minimap Widget|Autoinitialization", meta=(EditCondition="bAutoInitialized"))
	float AutoInitializeDelay;

	/** Determines whether the player's MinimapElementComponent is automatically added to the minimap when the widget is initialized.
	 * If true, the player's icon will be added to the minimap without requiring previous component setup.*/
	UPROPERTY(EditAnywhere, Category="Minimap Widget")
	bool bAutoAddPlayerIconToMinimap;

	/**
	 * Initialize the minimap and all of its components. If bAutoInitialized this method will be called automatically
	 * @return True if initialization completed successfully
	 */
	UFUNCTION(BlueprintCallable, Category="Minimap Widget")
	bool InitializeMinimap();

	/**
	 * Set the minimap icon position inside of the minimap, relative to the Minimap Component placed in 3D world
	 * @param MinimapIcon - Minimap Icon that will be updated
	 * @param MinimapComponent - Minimap Component that will provide the world location
	 */
	UFUNCTION(BlueprintCallable, Category="Minimap Widget")
	void SetMinimapIconPosition(UUserWidget* MinimapIcon, UMinimapElementComponent* MinimapComponent);

	/**
	 * Used to find the pawn controlled for the player and add a minimap icon in its position.
	 */
	UFUNCTION(BlueprintCallable, Category="Minimap Utilities", meta=(WorldContext="WorldContext"))
	void AddPlayerIconToMinimap();

	/**
	 * Update the minipap widget asociated with the provided Minimap Component
	 * @param IconToUpdate - Static Icon that will be updated
	 */
	UFUNCTION(BlueprintCallable, Category="Minimap Widget")
	void UpdateStaticMinimapIcon(UMinimapElementComponent* IconToUpdate);

	UFUNCTION(BlueprintPure, Category="Minimap Widget")
	UMinimapConfigAsset* GetMinimapConfiguration() const;
	
	UFUNCTION(BlueprintCallable, Category="Minimap Widget")
	void UpdateMinimapForLevel(int32 LevelIndex);
	UFUNCTION(BlueprintPure, Category="Minimap Widget")
	int32 GetCurrentLevelIndex() const;
	UFUNCTION()
	void OnLevelChanged(int32 NewLevelIndex);
	
	protected:
	virtual void NativePreConstruct() override;
	virtual void NativeConstruct() override;
	virtual void NativeOnInitialized() override;
	virtual void NativeTick(const FGeometry& MyGeometry, float InDeltaTime) override;
	void InitMinimapSize(const FVector2D& InMinimapSize) const;
	
	bool GetFloorConfig(int32 LevelIndex, FMinimapFloorData& MinimapFloor) const;
	void UpdateMinimapWorldBounds();

	void ClearMinimapIcons();
	
	UFUNCTION(BlueprintCallable, Category="Minimap Widget")
	TArray<FMinimapFloorData> GetMinimapFloors() const;

private:
	FMinimapWidgetSettings MinimapWidgetSettings;
	
	float MinimapWorldSize;
	FVector WorldCameraLocation;
	FVector2D MinimapWorldLimits;
	FVector2D MinimapWidgetLimits;
	
	UPROPERTY() TObjectPtr<UMaterialInstanceDynamic> MinimapMaterialInstance;
	UPROPERTY() TArray<TObjectPtr<UUserWidget>> MinimapIconWidgetsA;
	UPROPERTY() TMap<UMinimapElementComponent*, UUserWidget*> DynamicMinimapIconWidgets;
	UPROPERTY() TMap<UMinimapElementComponent*, UUserWidget*> StaticMinimapIconWidgets;
	UPROPERTY()	TObjectPtr<UMinimapConfigAsset> MinimapConfig;
	
	UFUNCTION() bool InitMinimapRenderComponents();
	UFUNCTION() bool InitMinimapVisualComponents();
	UFUNCTION() void CreateMinimapItems();
	UFUNCTION() bool CreateMinimapIconWidgetFromMinimapComp( UMinimapElementComponent* MinimapComponent);
	UFUNCTION() void OnIconDynamicChanged(UMinimapElementComponent* MinimapComponent);
};
