// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "GameplayTagContainer.h"
#include "Components/ActorComponent.h"
#include "Components/TimelineComponent.h"
#include "GameFramework/SpringArmComponent.h"
#include "ITwISPFocusInteractComponent.generated.h"

class UCameraComponent;
class UIdealTwinViewConfig;
DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnFocusTransitionFinish);


/**
 * A component that handles camera focus transitions between positions in the world.
 * Extends spring arm functionality to provide smooth camera movement for interactive spots.
 * Used to focus the camera on points of interest with customizable transition effects.
 */
UCLASS(DisplayName="FocusInteractComponent",ClassGroup=("IdealTwin"), meta=(BlueprintSpawnableComponent))
class IDEALTWINPRO_API UITwISPFocusInteractComponent : public USpringArmComponent
{
	GENERATED_BODY()

public:
	UITwISPFocusInteractComponent(const FObjectInitializer& ObjectInitializer);

	/**
	 * Event triggered when a focus transition animation completes.
	 * Can be used to trigger additional effects or logic after the camera has reached its target position.
	 */
	UPROPERTY(BlueprintAssignable, Category = "FocusInteractComponent")
	FOnFocusTransitionFinish OnFocusTransitionFinish;
	
	
	// This Id is use to search this focus component by the manager. Can be null.
	/**
	 * Unique identifier for this focus component.
	 * Used by the focus manager to reference and locate specific focus components in the world.
	 * Can be null if the component doesn't need to be referenced directly.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "FocusInteractComponent")
	FGameplayTag FocusComponentId;
	
	/**
	 * Configuration asset that defines how the camera should be positioned.
	 * Contains settings for field of view, distance, offset and other camera parameters.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Instanced, Category = "FocusInteractComponent")
	TObjectPtr<UIdealTwinViewConfig> ViewConfig;


	/**
	 * Determines whether to apply rotation transitions during focus.
	 * When true, the camera will smoothly rotate to the target rotation.
	 * When false, only position transitions will be applied.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "FocusInteractComponent")
	bool bApplyRotation;
	
	/**
	 * Animation curve that controls the transition timing.
	 * Defines how the camera moves between positions - can create effects like 
	 * ease-in/ease-out, bounce, etc. based on the curve shape.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "FocusInteractComponent")
	UCurveFloat* TransitionCurve;
	
	// This method can be override in BP
	/**
	 * Initiates the camera focus transition to this component.
	 * Starts the timeline animation that moves the camera to focus on this component's position.
	 * Can be overridden in Blueprints to customize the focus behavior.
	 */
	UFUNCTION(BlueprintCallable, BlueprintNativeEvent, Category = "FocusInteractComponent")
	void SetFocusToComponent();

#if WITH_EDITOR
	virtual void PostEditChangeProperty(FPropertyChangedEvent& PropertyChangedEvent) override;
	/** Instantiates and attaches a new camera component for preview view config.*/
	UFUNCTION(BlueprintCallable,CallInEditor,Category = "FocusInteractComponent|Editor",meta=(DevelopmentOnly))
	void PlaceCamera();
	/** Removes and destroys the camera component for preview view config.*/
	UFUNCTION(BlueprintCallable,CallInEditor,Category = "FocusInteractComponent|Editor",meta=(DevelopmentOnly))
	void RemoveCamera();
#endif
	
	
	virtual void TickComponent(float DeltaTime, enum ELevelTick TickType, FActorComponentTickFunction* ThisTickFunction) override;

protected:
	virtual void BeginPlay() override;

	UFUNCTION() virtual void OnUpdateProcess(float InAlpha);
	UFUNCTION() virtual void OnFinishTimeline();
	
private:

	UPROPERTY() FTimeline FocusTimeline;
	UPROPERTY() TObjectPtr<APlayerController> PC;
	UPROPERTY() TObjectPtr<APawn> FocusPawn;
	UPROPERTY() TObjectPtr<USpringArmComponent> SpringArm;
	UPROPERTY()	TArray<FVector> PositionsTrack;
	UPROPERTY()	TArray<FRotator> RotationsTrack;
	UPROPERTY()	TArray<float> TargetLenghtTrack;

	UPROPERTY()
	TObjectPtr<UCameraComponent> CameraComponent;
};
