// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "GameplayTagContainer.h"
#include "UObject/Object.h"
#include "ITwISPWorldSubsystem.generated.h"

class AITwInteractiveSpot;



DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnISPSelectionChanged, const TArray<AITwInteractiveSpot*>&, ISPSelection);
DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnISPPlayerLocationChanged, AITwInteractiveSpot*, ISPActor);


/**
 * Defines the different methods for filtering interactive spots by tags.
 * 
 * This enum specifies which type of tag association should be used when
 * filtering interactive spots for operations like visibility control.
 * Each filter type targets a different aspect of the interactive spot system.
 */
UENUM(BlueprintType)
enum EISPFilterType
{
	/** Filter by the ISP's group tag.
	 * Targets the primary categorization tag of the interactive spot itself.
	 * Useful for filtering by location, area, or major functional category. */
	EFT_ISPGroup,
	
	/** Filter by the payload's group tag.
	 * Targets the main category tag of the payload attached to the interactive spot.
	 * Useful for filtering by content type or information category. */
	EFT_PayloadGroup,
	
	/** Filter by any tag in the payload's tag container.
	 * Provides the most fine-grained filtering by matching any tag in the payload.
	 * Useful for complex filtering based on multiple payload attributes. */
	EFT_PayloadTag,
};

/**
 * Subsystem that manages interactive spots placement in the world.
 * 
 * The Interactive Spot Placement System (ISP) is a core component that centralizes 
 * the management of all interactive points throughout the game world. It serves as 
 * the coordination layer between interactive spots and the player's interactions with them.
 * 
 * This subsystem provides several key functions:
 * - Registration and tracking of all interactive spots in the world
 * - Selection and multi-selection handling for interactive spots
 * - Management of player location tracking through interactive spots
 * - Focus camera control for detailed viewing of interactive spots
 * - Visibility filtering based on tags and properties
 * 
 * Game systems can register for callbacks through the provided delegates to react
 * when selection states change or when the player moves between interactive locations.
 * This enables implementation of UI updates, gameplay events, or environmental changes
 * triggered by player movement between interactive areas.
 * 
 * The ISP system is designed to work seamlessly with the AITwInteractiveSpot class and its
 * derivatives to create a consistent interactive experience throughout the game.
 */
UCLASS(DisplayName="InteractSpotPlacementSystem")
class IDEALTWINPRO_API UITwISPWorldSubsystem : public UWorldSubsystem
{
	GENERATED_BODY()

	//~ Begin UWorldSubsystem interface
	virtual bool ShouldCreateSubsystem(UObject* Outer) const override;
	virtual void Initialize(FSubsystemCollectionBase& Collection) override;
	virtual void Deinitialize() override;
	virtual void OnWorldBeginPlay(UWorld& InWorld) override;
protected:
	virtual bool DoesSupportWorldType(const EWorldType::Type WorldType) const;
	//~ End UWorldSubsystem interface	


public:
	static UITwISPWorldSubsystem* Get(const UWorld* InWorld);

	
	/**
	 * Delegate that broadcasts when the selection of interactive spots changes.
	 * 
	 * This delegate is triggered whenever the set of selected interactive spots changes,
	 * whether through direct user interaction, programmatic selection, or clearing
	 * of the selection. It provides the complete array of currently selected ISP actors
	 * to allow UI elements or game systems to update their state accordingly.
	 * 
	 * Use cases include:
	 * - Updating UI panels to display information about selected spots
	 * - Triggering animations or visual effects on selected spots
	 * - Enabling context-specific controls based on the current selection
	 * 
	 * The delegate passes an array of spots rather than individual spots to support
	 * multi-selection scenarios efficiently.
	 */
	UPROPERTY(BlueprintAssignable,Category = "InteractSpotPlacementSystem")
	FOnISPSelectionChanged OnISPSelectionChanged;
	
	/**
	 * Delegate that broadcasts when the player's location ISP changes.
	 * 
	 * This delegate is triggered whenever the player moves from one interactive spot
	 * location to another. It provides the new ISP actor that the player is now located at,
	 * allowing game systems to react to the player's movement between significant locations.
	 * 
	 * Player location tracking is distinct from selection - a player can be physically
	 * located at one spot while having a different spot (or multiple spots) selected.
	 * 
	 * Common use cases include:
	 * - Triggering area-specific gameplay effects when entering a new location
	 * - Updating navigation systems or minimaps to reflect current position
	 * - Playing ambient sounds or environmental effects based on location
	 * - Activating storyline events or character dialogues tied to specific locations
	 * 
	 * When the player leaves all interactive spots, this delegate is broadcast with a nullptr.
	 */
	UPROPERTY(BlueprintAssignable,Category = "InteractSpotPlacementSystem")
	FOnISPPlayerLocationChanged OnISPPlayerLocationChanged;
	
	/**
	 * Shows or hides all interactive spot actors in the world simultaneously.
	 * 
	 * This function provides a global control for the visibility of all registered
	 * interactive spots, which is useful for scenario transitions, UI modes, or gameplay
	 * state changes that require temporarily hiding or revealing all interactive elements.
	 * 
	 * The visibility affects both the visual components (meshes, widgets) and the
	 * interactive capabilities of the spots. Hidden spots cannot be selected or interacted with.
	 * 
	 * Note that this function overrides any individual visibility settings that might
	 * have been applied to spots previously. When making spots visible again, they
	 * return to their default visibility states.
	 * 
	 * @param bVisible - True to make all spots visible, false to hide all spots
	 */
	UFUNCTION(BlueprintCallable, Category = "InteractSpotPlacementSystem")
	void VisibilityAllISPActor(bool bVisible);
	
	/**
	 * Filters interactive spots by tags and selectively controls their visibility.
	 * 
	 * This function provides fine-grained control over which interactive spots are
	 * visible based on their associated tags. It supports three different filtering
	 * modes that target different aspects of the interactive spots:
	 * 
	 * - ISP Group: Filters based on the spot's group tag, useful for showing/hiding
	 *   spots that belong to specific areas or functional categories.
	 * 
	 * - Payload Group: Filters based on the payload's group tag, allowing control
	 *   of spots based on the type of content they provide.
	 * 
	 * - Payload Tag: Filters based on any tag in the payload's tag container, enabling
	 *   fine-grained filtering based on specific payload attributes.
	 * 
	 * This granular filtering system allows for complex visibility scenarios, such as:
	 * - Revealing only points of interest relevant to current objectives
	 * - Toggling between different types of interactive elements (info points vs. teleporters)
	 * - Progressive disclosure of interactive spots as the player advances
	 * 
	 * @param Tags - The set of gameplay tags to filter by (spots matching ANY tag will be affected)
	 * @param bVisible - Whether the filtered spots should be made visible (true) or hidden (false)
	 * @param FilterType - The type of filtering to apply (group-based, payload-based, etc.)
	 */
	UFUNCTION(BlueprintCallable,DisplayName="Visibility ISP By Filter" ,Category = "InteractSpotPlacementSystem")
	void VisibilityISPByFilter(FGameplayTagContainer Tags, bool bVisible,TEnumAsByte<EISPFilterType> FilterType);

	
	/**
	 * Enables or disables multi-selection mode for interactive spots.
	 * 
	 * The selection mode determines how interactive spots respond when a new spot is selected:
	 * 
	 * - In single selection mode (default), selecting a new spot will deselect any
	 *   previously selected spots, maintaining only one selected spot at a time.
	 *   This is appropriate for focused interactions where the player should only
	 *   work with one interactive element at a time.
	 * 
	 * - In multi-selection mode, selecting a new spot adds it to the current selection
	 *   without affecting previously selected spots. This allows building up a set of
	 *   selected spots for batch operations or comparison.
	 * 
	 * Multi-selection is particularly useful for:
	 * - Comparing information between multiple points of interest
	 * - Setting up linked operations between multiple interactive spots
	 * - Creating custom routes or sequences of interactive spots
	 * 
	 * The current selection mode affects all subsequent selection operations until changed.
	 * 
	 * @param bActiveMultiSelection - True to enable multi-selection mode, false for single selection mode
	 */
	UFUNCTION(BlueprintCallable, Category = "InteractSpotPlacementSystem")
	void SetMultiSelection(bool bActiveMultiSelection);
	
	/**
	 * Checks if multi-selection mode is currently active.
	 * @return True if multi-selection is enabled
	 */
	UFUNCTION(BlueprintCallable, Category = "InteractSpotPlacementSystem")
	bool IsMultiSelectionActive()const;

	/**
	 * Toggles the focusable state of all Interactive Spot (ISP) actors managed by the subsystem.
	 * 
	 * The focus property determines whether an interactive spot can trigger camera focusing
	 * when selected. This function provides a global override to enable or disable this
	 * functionality across all spots simultaneously, which is useful for changing interaction
	 * modes or accessibility options.
	 * 
	 * When focus capability is enabled:
	 * - Interactive spots with the EISPAP_Focus property will engage camera focus when selected
	 * - The camera will smoothly transition to view positions defined in each spot's ViewConfig
	 * - Player movement may be restricted while in focus mode
	 * 
	 * When focus capability is disabled:
	 * - Selecting spots will not trigger camera movement
	 * - The player maintains full camera control at all times
	 * - Any ongoing focus operations will be terminated
	 * 
	 * This setting affects the entire interactive spot system and takes precedence over 
	 * individual spot settings. It's particularly useful for temporarily changing the 
	 * interaction mode during special sequences or UI states.
	 * 
	 * @param bFocusable Determines whether the ISP actors should be focusable (true) or not (false)
	 */
	UFUNCTION(BlueprintCallable,DisplayName="Set All ISP Focusable" ,Category = "InteractSpotPlacementSystem")
	void SetAllISPFocusable(bool bFocusable);


	/**
	 * Saves the current player pawn's transform for later restoration.
	 * 
	 * This function captures the player's current position, rotation, and controller 
	 * orientation to enable returning to this exact state later. It's primarily used
	 * before initiating camera focus on an interactive spot, ensuring the player can
	 * return to their original position and orientation afterward.
	 * 
	 * The saved transform remains stored until explicitly overwritten by another call
	 * to this function or until the saved transform is restored through ResetPawnTransform().
	 * 
	 * Note: This function is typically called automatically by the focus system and
	 * rarely needs to be called directly unless implementing custom camera behaviors.
	 */
	void SavePawnTransform();
	
	/**
	 * Resets the pawn to its last saved transform before focusing.
	 * Used when exiting focus mode to return the player to their original position.
	 */
	UFUNCTION(BlueprintCallable, Category = "InteractSpotPlacementSystem")
	void ResetPawnTransform();
	
	
	/**
	 * Selects an interactive spot, updating its visual state and tracking it in the subsystem.
	 * 
	 * This function handles the selection logic for interactive spots, respecting the
	 * current selection mode (single or multi-selection) and triggering appropriate
	 * visual feedback and system notifications.
	 * 
	 * Selection behavior:
	 * - In single selection mode (default), any previously selected spots are deselected first
	 * - In multi-selection mode, the new spot is added to the existing selection
	 * - If the spot is already selected, this function has no effect
	 * - Only spots with the EISPAP_Selectable property can be selected
	 * 
	 * When a spot is selected:
	 * 1. Its visual state is updated through the OnChangeISPSelection event
	 * 2. It's added to the internal tracking array of selected spots
	 * 3. The OnISPSelectionChanged delegate is broadcast with the updated selection
	 * 
	 * Selection is a prerequisite for many interactive spot operations, including
	 * focusing the camera on the spot or triggering spot-specific actions.
	 * 
	 * @param InSelectSpot - The interactive spot to select
	 */
	UFUNCTION(BlueprintCallable, Category = "InteractSpotPlacementSystem")
	void SelectISP(AITwInteractiveSpot* InSelectSpot);
	
	/**
	 * Removes an interactive spot from the current selection.
	 * Updates the spot's visual state and removes it from tracking.
	 * @param InSelectSpot - The interactive spot to unselect
	 */
	UFUNCTION(BlueprintCallable, Category = "InteractSpotPlacementSystem")
	void UnSelectISP(AITwInteractiveSpot* InSelectSpot);
	
	/**
	 * Clears all currently selected interactive spots.
	 * Updates the visual state of all spots and resets the player's transform.
	 */
	UFUNCTION(BlueprintCallable, Category = "InteractSpotPlacementSystem")
	void ClearISPSelection();

	/**
	 * Sets an interactive spot as the player's current location.
	 * 
	 * This function updates the system's tracking of which interactive spot the player
	 * is currently located at. This tracking is separate from selection - a player can
	 * be physically located at one spot while selecting and interacting with others.
	 * 
	 * Player location tracking enables:
	 * - Location-aware gameplay events and triggers
	 * - Map and navigation updates based on player position
	 * - Area-specific environmental effects or UI states
	 * - Spatial analytics for understanding player movement patterns
	 * 
	 * When the player's location changes:
	 * 1. Any associated actions from the spot's ISPActions component are triggered
	 * 2. The OnISPPlayerLocationChanged delegate is broadcast with the new location
	 * 3. Game systems monitoring this delegate can update their state accordingly
	 * 
	 * This function is typically called automatically when a player overlaps with an
	 * interactive spot or its trigger volumes, but can also be called directly to
	 * programmatically set the player's logical location.
	 * 
	 * @param ISPActor - The interactive spot representing the player's new location
	 */
	UFUNCTION(BlueprintCallable, Category = "InteractSpotPlacementSystem")
	void SetISPPlayerLocation(AITwInteractiveSpot* ISPActor);
	
	/**
	 * Notifies that the player has left an interactive spot location.
	 * Updates tracking and broadcasts appropriate events.
	 * @param ISPActor - The interactive spot the player has left
	 */
	UFUNCTION(BlueprintCallable, Category = "InteractSpotPlacementSystem")
	void PlayerLeaveLocationISP(AITwInteractiveSpot* ISPActor);
	
	/** Gets all interactive spot actors registered with this subsystem.
	 * @return Array of all interactive spot actors
	 */
	UFUNCTION(BlueprintPure, Category = "InteractSpotPlacementSystem")
	TArray<AITwInteractiveSpot*> GetISPActors();

	UFUNCTION(BlueprintPure, Category = "InteractSpotPlacementSystem")
	TArray<AITwInteractiveSpot*> GetISPActorsByFilter(FGameplayTagContainer Tags,TEnumAsByte<EISPFilterType> FilterType);

	

	/** Retrieves interactive spot actors based on specified group tags.
	 * It is used for operations that require filtering of interactive spot
	 * actors by gameplay group associations.
	 *
	 * @param GroupTags The gameplay tag container used to filter interactive spot actors by group association.
	 * @return An array of interactive spot actors that match any of the specified group tags.
	 */
	UFUNCTION(BlueprintCallable, Category = "InteractSpotPlacementSystem")
	TArray<AITwInteractiveSpot*> GetISPActorsByGroup(FGameplayTagContainer GroupTags);

	
	/** Gets all currently selected interactive spot actors.
	 * @return Array of selected interactive spot actors
	 */
	UFUNCTION(BlueprintPure, Category = "InteractSpotPlacementSystem")
	TArray<AITwInteractiveSpot*> GetISPSelectionActors();
	
private:
	bool bMultiSelection;

	FTransform PawnLastNoFocusTransform;
	FRotator LastPawnControlRotation;
	UPROPERTY() TObjectPtr<AITwInteractiveSpot> ISPPlayerLocationActor;
	// All actors
	UPROPERTY() TArray<TObjectPtr<AITwInteractiveSpot>> ISPActors;
	// Selected Actors
	UPROPERTY() TArray<TObjectPtr<AITwInteractiveSpot>> ISPSelection;
};

