// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "GameplayTagContainer.h"
#include "GameFramework/Actor.h"
#include "InteractSpotPlacementActions/ITwISPActionsBase.h"
#include "ITwInteractiveSpot.generated.h"

class UIdealTwinViewConfig;
class UITwISPPayLoadList;
class UITwISPActionsBase;
class UWidgetComponent;
class UITwISPPayLoad;
class UITwISPFocusInteractComponent;
class UITwISPWidget;
class ATriggerBase;

/**
 * Defines the method for constructing payloads for interactive spots.
 * This enum determines whether the interactive spot's payload is loaded
 * from a predefined list or created as a unique instance within the spot.
 */
UENUM(BlueprintType)
enum EISPPayloadConstruct
{
	/** Load the payload from a predefined list using a matching tag.
	 * This mode is useful when reusing the same payload across multiple spots. */
	EISPC_FromList		UMETA(DisplayName = "From List"),
	
	/** Create a unique payload instance for this specific interactive spot.
	 * This mode allows customizing the payload directly on the spot. */
	EISPC_FromInstance	UMETA(DisplayName = "From Instance"),
	
};




UENUM(BlueprintType, meta = (Bitflags, UseEnumValuesAsMaskValuesInEditor = "true"))
enum EISPActorProperties
{
	EISPAP_None				= 0 UMETA(Hidden),
	EISPAP_Selectable		= 1 << 0 UMETA(DisplayName = "Selectable",Tooltip="Can be selected"),
	EISPAP_Focus			= 1 << 1 UMETA(DisplayName = "Focus On Selection",Tooltip="Can be focused when the ISP is selected"),
	EISPAP_3DWidget			= 1 << 2 UMETA(DisplayName = "3D Widget", Tooltip="Display a 3D widget from the payload"),
	EISPAP_VisualVolume		= 1 << 3 UMETA(DisplayName = "Visual Volume",Tooltip="Display a staticmesh for visual volume for the ISP"),
	EISPAP_PawnOverlap		= 1 << 4 UMETA(DisplayName = "Pawn Overlap",Tooltip="Capture when pawn enters or exits the volume"),
	EISPAP_ActorClick		= 1 << 5 UMETA(DisplayName = "Actor Click",Tooltip="Capture Actor Clicks"),
	EISPAP_All				= EISPAP_Selectable | EISPAP_Focus |EISPAP_3DWidget | EISPAP_VisualVolume | EISPAP_PawnOverlap | EISPAP_ActorClick UMETA(DisplayName = "All"),
};
ENUM_CLASS_FLAGS(EISPActorProperties);
ENUM_RANGE_BY_VALUES(EISPActorProperties, 
	EISPActorProperties::EISPAP_3DWidget,
	EISPActorProperties::EISPAP_VisualVolume,
	EISPActorProperties::EISPAP_PawnOverlap,
	EISPActorProperties::EISPAP_ActorClick,
	);



/**
 * AITwInteractiveSpot - Base class for interactive points in the world
 * 
 * This class represents interactive spots in the game world that players can interact with.
 * Interactive spots can display 3D widgets, respond to selection, focus the camera on points of interest,
 * detect when players enter or leave their volume, and respond to clicks.
 * 
 * The class supports a flexible payload system that can store and provide custom data
 * for each interactive spot, and handles various interaction modes including:
 * - Selection (clicking or interacting with the spot)
 * - Volume overlap (entering or exiting the spot's area)
 * - Camera focus (moving and orienting the camera to view the spot)
 * - Widget display (showing relevant UI elements in 3D space)
 * 
 * Interactive spots can be configured with different visual representations and
 * behavior through the ISPActorProperties flags.
 */
UCLASS(DisplayName="Interactive Spot",PrioritizeCategories=("Interact Spot"))
class IDEALTWINPRO_API AITwInteractiveSpot : public AActor
{
	GENERATED_BODY()
public:
	AITwInteractiveSpot(const FObjectInitializer& ObjectInitializer);

	/** Determines if the interactive spot is visible when the game starts.
	 * When true, the spot will be visible to the player immediately upon game start.
	 * When false, the spot will be hidden until explicitly made visible through code. */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Interact Spot")
	bool bStartVisible;
	
	/** The gameplay tag that identifies this interactive spot's group.
	 * Used for organizing and filtering interactive spots into logical categories.
	 * For example, spots can be grouped by area or functionality type. */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Interact Spot")
	FGameplayTag ISPGroup;
	
	/** Configures the behavior and appearance properties of this interactive spot.
	 * This bitmask controls various aspects like selection behavior, visual representation,
	 * and interaction methods. See EISPActorProperties for individual options. */
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="Interact Spot", meta = (Bitmask, BitmaskEnum = "/Script/IdealTwinPro.EISPActorProperties"))
	int32 ISPActorProperties;
	

	
	/** Determines the method used to construct the payload for the interactive spot.
	 * It defines whether the payload is selected from a predefined list or created as an individual instance.
	 * Useful for configuring and managing payload behaviors in interactive spots. */
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category = "Interact Spot|ISP Payload")
	TEnumAsByte<EISPPayloadConstruct> PayloadConstructMode;
	
	/** Identifies which payload to use from the PayLoadList when PayloadConstructMode is set to FromList.
	 * This tag should match the PayloadID of an item in the associated PayLoadList. */
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="Interact Spot|ISP Payload",meta=(EditCondition="PayloadConstructMode == EISPPayloadConstruct::EISPC_FromList", EditConditionHides))
	FGameplayTag PayloadIDInList;
	
	/** The list of available payloads to choose from when PayloadConstructMode is set to FromList.
	 * Contains a collection of payload objects that can be assigned to this interactive spot based on PayloadIDInList. */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Interact Spot|ISP Payload",meta=(EditCondition="PayloadConstructMode == EISPPayloadConstruct::EISPC_FromList", EditConditionHides))
	TObjectPtr<UITwISPPayLoadList> PayLoadList;
	
	/** The direct payload instance to use when PayloadConstructMode is set to FromInstance.
	 * Allows creating a unique payload configuration directly within this interactive spot. */
	UPROPERTY(EditAnywhere,Instanced, Category = "Interact Spot|ISP Payload",meta=(EditCondition="PayloadConstructMode == EISPPayloadConstruct::EISPC_FromInstance", EditConditionHides))
	TObjectPtr<UITwISPPayLoad> InstancedPayload;
	
	/** Defines the action class that will handle this interactive spot's behavior.
	 * This class will be instantiated at runtime to provide custom functionality
	 * like special effects, animations, or game logic when the spot is interacted with. */
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category = "Interact Spot")
	TSoftClassPtr<UITwISPActionsBase> ISPActions;
	
	/** An array of trigger volumes associated with the interactive spot.
	 * Used for setting up overlap events and checking conditions like whether the player's pawn
	 * is overlapping with any of these trigger volumes. */
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category = "Interact Spot")
	TArray<ATriggerBase*> ISPTriggerVolumes;
	
	
	/** Sets a new payload for this interactive spot.
	 * @param InPayload - The payload object to assign to this spot.
	 * 
	 * This function updates the payload and any associated data or visuals.
	 * For example, if the payload contains widget references, the 3D widget will be updated. */
	UFUNCTION(BlueprintCallable, Category = "Interact Spot")
	void SetPayload(UITwISPPayLoad* InPayload);
	
	/** Event called when the selection state of this interactive spot changes.
	 * @param bSelected - True if the spot has been selected, false if deselected.
	 * 
	 * Override this function in Blueprint to implement custom visual feedback
	 * or behavior when the spot is selected or deselected by the player. */
	UFUNCTION(BlueprintNativeEvent, Category = "Interact Spot")
	void OnChangeISPSelection(bool bSelected);


	
	UFUNCTION(BlueprintImplementableEvent, Category = "Interact Spot")
	void OnPayloadLoaded(const UITwISPPayLoad* InPayload);
	
	/** Checks if this interactive spot is currently selected.
	 * @return True if the spot is selected, false otherwise.
	 * 
	 * This function can be used to conditionally execute logic based on
	 * whether the interactive spot is in a selected state. */
	UFUNCTION(BlueprintCallable, Category = "Interact Spot")	
	bool IsISPSelected()const;
	
	/** Selects this interactive spot if it's selectable.
	 * 
	 * This function will attempt to select the spot, updating its visual state
	 * and notifying the ISP subsystem. The selection will only occur if
	 * the EISPAP_Selectable flag is set in ISPActorProperties. */
	UFUNCTION(BlueprintCallable, Category = "Interact Spot")	
	void SelectISP();
	
	/** Deselects this interactive spot if it's currently selected.
	 * 
	 * This function will attempt to deselect the spot, reverting its visual state
	 * and notifying the ISP subsystem. The deselection will only occur if
	 * the spot is currently selected and the EISPAP_Selectable flag is set. */
	UFUNCTION(BlueprintCallable, Category = "Interact Spot")	
	void UnSelectISP();
	
	/** Gets the distance between a component and the player's camera.
	 * @param Component - The component to measure distance from.
	 * @return The distance in unreal units between the component and the player, or -1 if player not found.
	 * 
	 * Useful for implementing distance-based behavior like LOD or visibility toggling. */
	UFUNCTION(BlueprintCallable, Category = "Interact Spot")	
	float GetDistanceToCamera(USceneComponent* Component)const;
	
	/** Checks if there's a clear line of sight between a component and the player.
	 * @param Component - The component to check visibility from.
	 * @param Channel - The trace channel to use for the visibility check.
	 * @param Radius - The radius of the sphere trace used for the check.
	 * @return True if the player is visible from the component, false otherwise.
	 * 
	 * This function performs a sphere trace between the component and player to determine visibility.
	 * Useful for implementing visibility-dependent behavior or UI elements. */
	UFUNCTION(BlueprintCallable, Category = "Interact Spot")	
	bool CheckVisionLineWithPawn(USceneComponent* Component, TEnumAsByte<ETraceTypeQuery> Channel, float Radius= 20.f);
	
	virtual void NotifyActorOnClicked(FKey ButtonPressed = EKeys::LeftMouseButton) override;
	virtual void NotifyActorBeginOverlap(AActor* OtherActor) override;
	virtual void NotifyActorEndOverlap(AActor* OtherActor) override;

	/** Binds trigger volumes to the corresponding overlap events. */
	void BindTriggerVolumes();
	UFUNCTION() void NotifyTriggerVolumeBeginOverlap(AActor* OverlappedActor, AActor* OtherActor);
	UFUNCTION() void NotifyTriggerVolumeEndOverlap(AActor* OverlappedActor, AActor* OtherActor);
	/** Checks if the player's pawn is currently within any of the actor's trigger volumes.
	 * Iterates through the array of ISPTriggerVolumes, checking if the player's pawn
	 * is overlapping with any of them.
	 * @return True if the player's pawn is inside at least one of the trigger volumes, otherwise false.
	 */
	bool CheckForPawnInTriggerVolume();
	/** Registers this interactive spot as the player's current location.
	 * 
	 * This function notifies the ISP subsystem that the player is currently at this spot
	 * and triggers any associated actions via the ISPActions component.
	 * Typically called automatically when the player overlaps with the spot or its trigger volumes. */
	UFUNCTION(BlueprintCallable, Category = "Interact Spot")
	void SetAsPlayerLocationISP();
	
	/** Notifies that the player has left this interactive spot location.
	 * 
	 * This function informs the ISP subsystem that the player is no longer at this spot
	 * and triggers any exit actions via the ISPActions component.
	 * Typically called automatically when the player stops overlapping with the spot or its trigger volumes. */
	UFUNCTION(BlueprintCallable, Category = "Interact Spot")
	void PlayerLeaveLocationISP();
	
	/** Initiates camera focus on this interactive spot.
	 * 
	 * When called, this function uses the FocusInteractComponent to move and orient
	 * the camera to focus on this spot. The camera positioning is determined by
	 * the ViewConfig settings. Requires the EISPAP_Focus flag to be set. */
	UFUNCTION(BlueprintCallable, Category = "Interact Spot")
	void FocusISPActor();
	
	/** Retrieves the payload associated with this interactive spot.
	 * @return The payload object currently assigned to this spot, or nullptr if none is assigned.
	 * 
	 * The payload contains data specific to this interactive spot, such as widget classes,
	 * display information, or custom gameplay data. */
	UFUNCTION(BlueprintCallable, Category = "Interact Spot")
	UITwISPPayLoad* GetPayloadFromSpot()const;
	
	void LoadPayloadFromConstruct();

	UFUNCTION(BlueprintCallable, Category = "Interact Spot")
	void SetViewConfig(UIdealTwinViewConfig* InViewConfig);
	UFUNCTION(BlueprintCallable, Category = "Interact Spot")
	UIdealTwinViewConfig* GetViewConfig()const;


	
	/** Sets the payload ID to use when constructing a new payload instance.
	 * @param InPayloadID - The payload ID to use when constructing a new payload instance.
	 * 
	 * This function is used to set the payload ID when constructing a new payload instance
	 * from the PayLoadList. It is useful for creating a unique payload for this spot
	 * without having to manually assign it to the spot. */
	
#if WITH_EDITOR
	virtual void PostEditChangeProperty(FPropertyChangedEvent& PropertyChangedEvent) override;
	virtual void PostEditImport() override;
	
	UActorComponent* AddComponentToBP(UObject* Object,UActorComponent* Component);




	/** Instantiates and attaches a new camera component for preview view config.
	* This editor-only function creates a temporary camera to preview how the
	* interactive spot will look when focused on during gameplay. */
	UFUNCTION(BlueprintCallable,CallInEditor,Category = "Interact Spot")
	void PlaceCamera();
	
	/** Removes and destroys the camera component for preview view config.
	 * This editor-only function cleans up the temporary camera used for
	 * previewing the spot's focus view. */
	UFUNCTION(BlueprintCallable,CallInEditor,Category = "Interact Spot")
	void RemoveCamera();
#endif

	virtual void Tick(float DeltaTime) override;

	
	/** Configuration for camera positioning when focusing on this interactive spot.
	 * Defines parameters like camera distance, angle, and transition speed
	 * when the spot is selected or focused on. */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Instanced,Category = "Interact Spot")
	TObjectPtr<UIdealTwinViewConfig> ViewConfig;
	
	
	/** Component that handles camera focus functionality for this interactive spot.
	 * This component manages the positioning and transition of the camera when
	 * the spot is focused on. Only created if EISPAP_Focus flag is set. */
	UPROPERTY(BlueprintReadOnly, VisibleDefaultsOnly, Category = "Interact Spot")
	TObjectPtr<UITwISPFocusInteractComponent> FocusInteractComponent;
	
	/** Component that renders a UI widget in 3D space for this interactive spot.
	 * This component displays the widget specified in the spot's payload.
	 * Only created if EISPAP_3DWidget flag is set. */
	UPROPERTY(BlueprintReadOnly, VisibleDefaultsOnly, Category = "Interact Spot")
	TObjectPtr<UWidgetComponent> WidgetComponent;
	
	/** Component that provides a visual representation of this interactive spot.
	 * This static mesh allows the spot to be visible in the world and define
	 * its collision volume. Only created if EISPAP_VisualVolume flag is set. */
	UPROPERTY(BlueprintReadOnly, VisibleDefaultsOnly, Category = "Interact Spot")
	TObjectPtr<UStaticMeshComponent> MeshComponent;
protected:	
	virtual void BeginPlay() override;
	
	virtual void InitializeWidget();
	void SetWidgetFromPayload() const;

private:
	void ChangeSelection(bool InSelected);
	void ConstructComponents(bool bIsConstructTime);
	void UpdateUseComponents() const;

	
	bool bISPSelected;
	
	friend class UITwISPWorldSubsystem;

	UPROPERTY() TObjectPtr<UITwISPPayLoad> PayLoad;
	UPROPERTY()TObjectPtr<UITwISPActionsBase> Actions;
};
