// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "Features/InteractSpotPlacement/ITwInteractiveSpot.h"
#include "Kismet/KismetSystemLibrary.h"
#include "UObject/Object.h"
#include "ITwISPTeleportPoint.generated.h"

class UMinimapElementComponent;


USTRUCT(BlueprintType)
struct FISPTeleportVisionData
{
	GENERATED_BODY()
	
	/** Defines the offset distance used to determine if the teleport point is within the player's line of sight.
	 * Higher values make it easier to detect the teleport point in the field of view.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "TeleportVisionData")
	float VisionOffset;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "TeleportVisionData")
	TEnumAsByte<ETraceTypeQuery> VisionChannel;
	
	/** Controls the display of debug lines to check the visibility of the point.
	 * Useful for development to visualize the line traces used for visibility checks.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "TeleportVisionData")
	TEnumAsByte<EDrawDebugTrace::Type> DebugDrawLine;

	FISPTeleportVisionData()
		: VisionOffset(50.f)
		, VisionChannel(UEngineTypes::ConvertToTraceType(ECC_Camera))
		, DebugDrawLine(EDrawDebugTrace::None)
	{}
};


/**
 * AITwISPTeleportPoint - Interactive teleport point in the world
 * 
 * This class represents an interactive teleport point that allows the player
 * to instantly move to its location. The point is visible on the minimap and shows
 * a widget in the world when it's in the player's field of vision.
 * 
 * The component integrates with the minimap system and provides visual effects
 * during teleportation such as configurable fades.
 */
UCLASS(DisplayName="ISP Teleport Point")
class IDEALTWINPRO_API AITwISPTeleportPoint : public AITwInteractiveSpot
{
	GENERATED_BODY()
public:
	AITwISPTeleportPoint(const FObjectInitializer& ObjectInitializer);

	/** Duration in seconds of the fade-in effect before teleportation.
	 * Controls how quickly the screen fades to black before teleporting the player.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ISP TeleportPoint|Teleport")
	float FadeInTime;
	
	/** Wait time in seconds after teleportation before showing the player.
	 * Creates a pause between the teleport action and the fade-out effect.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ISP TeleportPoint|Teleport")
	float DelayAfterTeleport;
	
	/** Duration in seconds of the fade-out effect after teleportation.
	 * Controls how quickly the screen fades back in after teleporting the player.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ISP TeleportPoint|Teleport")
	float FadeOutTime;
	
	/** Color used for the fade effect during teleportation.
	 * Typically black, but can be changed for special effects or branding purposes.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ISP TeleportPoint|Teleport")
	FLinearColor FadeColor;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ISP TeleportPoint|WidgetScreen",meta=(InlineEditConditionToggle))
	bool bUseVisionCheck;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ISP TeleportPoint|WidgetScreen", meta=(EditCondition="bUseVisionCheck"))
	FISPTeleportVisionData VisionCheck;

	/** Determines the minimum (X) and maximum (Y) distances to interpolate the widget size.
	 * Controls the distance range within which the widget will be visible and change its size.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ISP TeleportPoint|WidgetScreen|Size")
	FVector2D WidgetWorldDistance;
	
	/** Represents the fixed size on screen of the displayed widget.
	 *
	 * - The X component defines the minimum size when the widget is closer to the player.
	 * - The Y component defines the maximum size when the widget is farther away from the player.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ISP TeleportPoint|WidgetScreen|Size")
	FVector2D WidgetSize;

	//TODO bHiddeTeleportOnTP is a temporal fix for teleportation issues with widgets visibility.
	/** If true, hides the teleport point's widget during teleportation (from pre-teleport to teleport completed). */
	UPROPERTY(EditAnywhere, BlueprintReadWrite,DisplayName="Hide Teleport During TP", Category = "ISP TeleportPoint|Teleport")
	bool bHiddeTeleportOnTP = false;
	
	/**
	 * Gameplay tags used to filter which ISP entries should be hidden during teleportation.
	 * During pre-teleport the subsystem is called with these tags to hide matching ISPs, after teleport
	 * they are called again to restore visibility. Leave empty to apply the filter broadly.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite,DisplayName="Hidden TP Tags", Category = "ISP TeleportPoint|Teleport")
	FGameplayTagContainer HiddenTPTag;


	/** Event that fires when the cursor hovers over the teleport point.
	 * Use this to create hover effects for the teleport point.
	 * 
	 * @param bIsHovered - True if the cursor is over the point, false otherwise.
	 */
	UFUNCTION(BlueprintNativeEvent, Category = "ISPTeleportPoint")
	void OnTeleportPointHover(bool bIsHovered);


	/** Event that fires just before teleportation occurs.
	 * Can be used to initiate visual or sound effects prior to teleportation,
	 * or to prepare the scene at the destination point.
	 */
	UFUNCTION(BlueprintNativeEvent, Category = "ISPTeleportPoint")
	void OnPreTeleported();
	virtual void OnPreTeleported_Implementation();
	
	/** Event that fires when the player is teleported.
	 * Executes right when the character has been moved to the new location,
	 * before any fade-out effects are complete.
	 */
	UFUNCTION(BlueprintNativeEvent, Category = "ISPTeleportPoint")
	void OnTeleported();
	virtual void OnTeleported_Implementation(){};
	
	/** Event that fires when the teleportation process has completely finished.
	 * This occurs after all fade effects and configured delays have finished.
	 * Use this for post-teleport game logic that should happen after the player can see again.
	 */
	UFUNCTION(BlueprintNativeEvent, Category = "ISPTeleportPoint")
	void OnTeleportCompleted();
	virtual void OnTeleportCompleted_Implementation();

	
	virtual void Tick(float DeltaTime) override;

	/** Initiates the process of teleporting the player to this point.
	 * Applies fade effects and activates teleportation events in sequence.
	 * Call this function to manually trigger the teleportation effect.
	 */
	UFUNCTION(BlueprintCallable, Category = "ISPTeleportPoint")
	void TriggerTeleport();

#if WITH_EDITOR
	virtual void PostEditChangeChainProperty(FPropertyChangedChainEvent& PropertyChangedEvent) override;
#endif
	
	
protected:
	/** Function called when interacting with the point's icon on the minimap.
	 */
	UFUNCTION()
	void OnIconElementInteract();
	
	virtual void BeginPlay() override;
	
	/** Adjusts the widget size on screen based on the distance to the player.
	 */
	void SetWidgetDrawSizeByDistance() const;
	
	/** Checks if the point is in the player's field of vision.
	 * @param InOffset - Offset for visibility check rays.
	 * @param DebugLine - Debug visualization type for rays.
	 * @return True if the point is visible to the player, false otherwise.
	 */
	bool CheckVisionWithPawn(float InOffset = 50.f, EDrawDebugTrace::Type DebugLine = EDrawDebugTrace::None) const;



protected:
	/** Component that allows the teleport point to appear on the minimap.
	 * Enables interaction through the minimap to activate teleportation.
	 */
	UPROPERTY(VisibleDefaultsOnly, BlueprintReadOnly, Category = "ISPTeleportPoint", meta = (AllowPrivateAccess = "true") )
	UMinimapElementComponent* MinimapElementComponent;

};
