#pragma once


#include "CoreMinimal.h"
#include "IdealTwinLevelVariantSetsActor.h"
#include "Engine/DataTable.h"
#include "IdealTwinVariantHandler.generated.h"


class UVariantSet;
class ULevelVariantSets;

typedef TMap<FName, TArray<FName>> WrapperSubCategories;
typedef TMap<FName, WrapperSubCategories> WrapperCategories;

DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnVariantChangeNotification);
DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnConfigurationChangesListUpdated);

/* This class is used to parse and manage the variant info and display it on UI*/
UCLASS(BlueprintType) class IDEALTWINPRO_API UIdealTwinVariantHandler : public UObject
{
	GENERATED_BODY()
public:
	UIdealTwinVariantHandler(const FObjectInitializer& ObjectInitializer);

	
	/*	If it's true every time the list change will be sent to database directly. False it should be called manually
		with SendConfiguratorList()
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "IdealTwinVariantHandler")
	bool bSendDataOnChange;


	UPROPERTY(BlueprintAssignable)
	FOnVariantChangeNotification OnVariantChangeNotification;
	UPROPERTY(BlueprintAssignable)
	FOnConfigurationChangesListUpdated OnConfigurationChangesListUpdated;

	/// INITIALIZATION
	UFUNCTION(BlueprintCallable, Category="VariantHandler")
	bool InitializedVariantSet(AIdealTwinLevelVariantSetsActor* OwnerVariant);
	

	
	///------------ITEM OVERRIDE DATA TABLE ---------
	UFUNCTION(BlueprintCallable, Category= "VariantHandler|DataTableReference")
	TArray<FName> GetVariantOverrideValues(const TArray<FName>& InValues);
	UFUNCTION(BlueprintCallable, Category= "VariantHandler|DataTableReference")
	bool GetVariantItemOverrideDisplay(FName RowName, FIdealTwinItemOverrideData& OutCategoryStruct);
	UFUNCTION(BlueprintCallable,Category="VariantHandler|DataTableReference")
	UObject* GetVariantImageImageOverride(const FName& RowName);

	TArray<FIdealTwinItemOverrideData> GetOverrideList(const TArray<FName>& InListNames);

	TArray<FIdealTwinItemOverrideData> GetAllCategoryOverrideData();
	TArray<FIdealTwinItemOverrideData> GetSubCategoryOverrideData();
	TArray<FIdealTwinItemOverrideData> GetProductOverrideData();	

	UFUNCTION(BlueprintCallable, Category = "VariantHandler")
	void SetCategory(const FName& Category);
	UFUNCTION(BlueprintCallable, Category = "VariantHandler")
	void SetSubCategory(const FName& SubCategory);
	UFUNCTION(BlueprintCallable, Category = "VariantHandler")
	bool SetVariant(const FName& Variant, bool bApply = false);

	UFUNCTION(BlueprintCallable, Category = "VariantHandler")
	void ClearSelections();

	UFUNCTION(BlueprintPure, Category = "VariantHandler")
	TArray<FName> GetAllCategoryValues()const;
	
	UFUNCTION(BlueprintPure, Category = "VariantHandler")
	TArray<FName> GetSubCatsFromSelectedCat(bool bUseItemOverride=true);
	UFUNCTION(BlueprintPure, Category = "VariantHandler")
	TArray<FName> GetSubCatsFromCat(const FName& CatKey, bool bUseItemOverride = true);
	UFUNCTION(BlueprintPure, Category = "VariantHandler")
	TArray<FName> GetVariantsFromSelectedSubCat(bool bUseItemOverride=true);
	UFUNCTION(BlueprintPure, Category = "VariantHandler")
	TArray<FName> GetVariantsFromSubCat(const FName& CatKey, const FName& SubCatKey, bool bUseItemOverride = true);

	
	/** Return all selected values 
	 */
	UFUNCTION(BlueprintPure,Category = "VariantHandler")
	void GetVariantSelectionData(FName& Category, FName& SubCategory, FName& Variant)const;

	
	UFUNCTION(BlueprintCallable, Category = "VariantHandler")
	FString GetSelectionOnSingleString()const;

	/// GET VARIANT ACTIVATED
	UFUNCTION(BlueprintCallable, Category = "VariantHandler")
	FName GetVariantActivated(FName Category, FName SubCategory);
	UFUNCTION(BlueprintCallable, Category = "VariantHandler")
	FName GetVariantActivatedFromSelection();
	UFUNCTION(BlueprintPure, Category = "VariantHandler")
	int32 HasIndexActivate(FName Category, FName SubCategory) const;


	/// APPLY TO VARIANT
	bool ApplyVariantByName(const FString& InName, ULevelVariantSets* InLevelVariantSets) const;
	UFUNCTION(BlueprintCallable, Category = "VariantHandler")
	bool ApplyVariantOption();
	bool ApplyVariantBySetNameAndProduct(const FString& InVariantSet, const FString& InProduct, ULevelVariantSets* InLevelVariantSets) const;

	UFUNCTION(BlueprintCallable, Category = "VariantHandler")
	bool GetVariantThumbnail(UTexture2D*& VariantThumbnail, FName VariantName);
	UFUNCTION(BlueprintPure, Category = "VariantHandler")
	bool GetVariantActiveIndex(int32& OutIndex);

	
	static FString CombineVariantSet(const FString& Cat, const FString& SubCat);

	UFUNCTION(BlueprintCallable, Category = "VariantHandler")
	bool GetProductDataFromVariant(const FName& VariantName, FIdealTwinConfigProduct& OutProduct);

	/// CONFIGURATOR LIST
	UFUNCTION(BlueprintCallable, Category = "VariantHandler|ConfiguratorChangesList")
	bool AddConfigurationChangeFromSelection();
	UFUNCTION(BlueprintCallable, Category = "VariantHandler|ConfiguratorChangesList")
	bool RemoveConfigurationChange(const FIdealTwinConfigProduct& ConfigProduct);
	
	UFUNCTION(BlueprintPure, Category = "VariantHandler|ConfiguratorChangesList")
	float GetConfiguratorListPrice()const;

	UFUNCTION(BlueprintCallable, Category = "VariantHandler|ConfiguratorChangesList")
	void SendConfiguratorList();
	UFUNCTION(BlueprintCallable, Category = "VariantHandler|ConfiguratorChangesList")
	void LoadConfigurationFromUser();

	UIdealTwinConfiguratorDataAsset* GetVariantConfigurator();

	UFUNCTION(BlueprintCallable, Category = "VariantHandler|ConfiguratorChangesList")
	TMap<FName,FIdealTwinConfigProduct> GetConfigtorChangeList() const;


	UFUNCTION(BlueprintCallable, Category = "VariantHandler")
	FName GetSelectedCategoryName()const {return SelectedCategory;}
	UFUNCTION(BlueprintCallable, Category = "VariantHandler")
	FName GetSelectedSubCategoryName()const	{return SelectedSubCategory;}
	UFUNCTION(BlueprintCallable, Category = "VariantHandler")
	FName GetSelectedProductName()const {return SelectedVariant;}
	
	
protected:
	FString GetFullCategoryString() const;

	void ParseVariantSetWrapper();
	bool SplitCategoriesFromSets(const UVariantSet& InVariantSet, FName& OutCategory, FName& OutSubCategory) const;
	bool SplitCompleteLine(const FString& InCompleteLine, TArray<FString>& OutValues)const;

	UFUNCTION() void Received_VariantConfiguratorUpdate();

	virtual void BeginDestroy() override;

private:
	FName SelectedCategory;
	FName SelectedSubCategory;
	FName SelectedVariant;

	/*Checker if Variant has all requisites*/
	bool bCategory, bSubCategory, bVariant;
	
	WrapperCategories CategoryWrapper;

	
	UPROPERTY() ULevelVariantSets* VariantSetReference;
	UPROPERTY() UIdealTwinConfiguratorItemOverride* ItemOverrideDisplay;

	UPROPERTY() UIdealTwinConfiguratorDataAsset* ConfiguratorAsset; 
};