// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "IdealTwinAPI/RealEstateAPI/Models/IdealTwinRealEstateDataModels.h"
#include "GameFramework/Actor.h"
#include "Kismet/KismetSystemLibrary.h"
#include "UnitVolumeActor.generated.h"


class UUnitVolumeAction;
class UUnitVolumeConfig;

USTRUCT(BlueprintType) struct IDEALTWINPRO_API FUnitVolumeUniqueIdentifier
{
	GENERATED_BODY()

	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="UnitVolumeData") int32 UnitVolumeID;
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="UnitVolumeData") FString UnitVolumeKeycode;
	FUnitVolumeUniqueIdentifier(): UnitVolumeID(-1)
	{}
	FUnitVolumeUniqueIdentifier(int32 InVolumeIndex,const FString& InVolumeKeycode):UnitVolumeID(InVolumeIndex),UnitVolumeKeycode(InVolumeKeycode){}
	FORCEINLINE bool operator == (const FUnitVolumeUniqueIdentifier& Other) const
	{
		return (UnitVolumeID == Other.UnitVolumeID) && (UnitVolumeKeycode == Other.UnitVolumeKeycode);
	}

	bool HasValidKeycode() const;
};

UENUM(BlueprintType) enum EUnitVolumeAvailability
{
	EHVA_UnValid		UMETA(DisplayName = "Unvalid"),
	EHVA_Available		UMETA(DisplayName = "Available"),
	EHVA_Reserved		UMETA(DisplayName = "Reserved"),
	EHVA_Sold			UMETA(DisplayName = "Sold"),
};


DECLARE_DYNAMIC_MULTICAST_DELEGATE_TwoParams(FOnUnitVolumesSelection, AUnitVolumeActor*,UnitVolume,bool, bSelected);


/**
	AUnitVolumeActor is the physical representation of the volumes that allows connecting the data from the idealTwin
	database to obtain the Unit information.
	This class is controlled by AUnitVolumeManager when making the selection and their visibility.

	They are designed to click on the volumes when they are visible and, once selected, return the data to be able to display them.

	This class implements the static CheckUnitVolumeUnderCursor() method that must be placed where the click logic is done.

	SETUP: To do the Unit volume setup, you have to instantiate them in the world, and once they are placed, assign the VolumeId.
	This ID can be found on the IdealTwin.com page in our UnitGroup. These Ids are relative to the UnitGroup.
	
 */
UCLASS()
class IDEALTWINPRO_API AUnitVolumeActor : public AActor
{
	GENERATED_BODY()

public:
	// Sets default values for this actor's properties
	AUnitVolumeActor(const FObjectInitializer& ObjectInitializer);

	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="UnitVolumeActor")
	FUnitVolumeUniqueIdentifier UnitVolumeIdentifier;
	/*Called when volume is selected or unselected*/
	UPROPERTY(BlueprintAssignable, Category="UnitVolumeScene")
	FOnUnitVolumesSelection OnUnitVolumeSelection;

	/** bExcludeVolumeFromManager is a flag that specifies whether a UnitVolumeActor is excluded from being managed by the AUnitVolumesManager.
	* When set to true, the UnitVolumeActor will not be added to the list of UnitVolumeScenes in the manager during initialization and refresh operations, effectively removing any management or interaction logic associated with the manager.
	* This property can be helpful for cases where specific UnitVolumeActors need to operate independently of the management system or follow custom behavior outside the established logic within AUnitVolumesManager.*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="UnitVolumeActor")
	bool bExcludeVolumeFromManager;
	
	/* If true the visibility of the volume will be visible at BeginPlay*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="UnitVolumeActor")
	bool bStartVisible;
	/* WARNING: This option will be overridden if the data is loaded from the database. In case this unit is not handled by manager will use this value*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="UnitVolumeActor")
	TEnumAsByte<EUnitVolumeAvailability> UnitAvailability;
	/* Use the general configuration of the Unit volume or use a custom one*/
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="UnitVolumeActor", meta=(InlineEditConditionToggle))
	bool bUseOverrideConfig;
	/* Visual configuration of the UnitVolumes such as Color Scheme*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="UnitVolumeActor", meta=(EditCondition="bUseOverrideConfig"))
	UUnitVolumeConfig* UnitVolumeConfiguration;

	/* Visual representation of how the volume of the Unit will be*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="UnitVolumeActor")
	UStaticMesh* UnitVolumeMesh;

	// Linking to a map to configure unit types by making transitions to interior maps
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="UnitVolumeActor")
	TSoftObjectPtr<UWorld> LinkerMap;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Instanced,Category = "UnitVolumeActor")
	TObjectPtr<UUnitVolumeAction> UnitVolumeAction;

	
	/* Implementation of a line trace using the mouse position on the screen, performing a deprojection on the screen to
		find the UnitVolumes underneath for subsequent interaction.
	*/
	UFUNCTION(BlueprintCallable, Category="UnitVolumeScene", meta=(WorldContext="WorldContext", AdvancedDisplay=4))
	static bool CheckUnitVolumeUnderCursor(const UObject* WorldContext, AUnitVolumeActor
	                                 *& SelectedUnitVolume, TEnumAsByte<ETraceTypeQuery> TraceChannel, float CheckDistance = 3000, EDrawDebugTrace
	                                 ::Type DrawDebugType =EDrawDebugTrace::None, float DrawLineTime =5);
	/* Linetrace function that detects if it has hit with a UnitVolumeScene*/
	UFUNCTION(BlueprintCallable, Category="UnitVolumeScene", meta=(WorldContext="WorldContext", AdvancedDisplay=5))
	static bool LineTraceUnitVolume(const UObject* WorldContext, const FVector& Start, const FVector& End, TEnumAsByte<ETraceTypeQuery> TraceChannel,
	                                AUnitVolumeActor*& SelectedUnitVolume, EDrawDebugTrace
	                                ::Type DrawDebugType, float DrawLineTime);

	/* Change volume visibility*/
	UFUNCTION(BlueprintCallable,Category="UnitVolumeScene")
	void SetUnitVolumeVisibility(bool bVisibility);
	UFUNCTION(BlueprintCallable,Category="UnitVolumeScene")
	bool IsUnitVolumeVisible()const { return !IsHidden(); }

	/* Select the volume. If bUseNotification is true it will call OnUnitVolumeSelection delegate*/
	UFUNCTION(BlueprintCallable, Category="UnitVolumeScene")
	void SelectUnitVolume(bool bUseNotification = false);
	/* Deselect the volume. If bUseNotification is true it will call OnUnitVolumeSelection delegate*/
	UFUNCTION(BlueprintCallable, Category="UnitVolumeScene")
	void UnSelectUnitVolume(bool bUseNotification= false);
	UFUNCTION(BlueprintCallable, Category="UnitVolumeScene")
	bool IsSelectedUnitVolume()const { return bSelectedUnitVolume; }

	/* Change the mesh of the visual representation of the volume*/
	UFUNCTION(BlueprintCallable, Category="UnitVolumeScene")
	void SetUnitVolumeStaticMesh(UStaticMesh* NewUnitMesh);

	/* Set the volume data. This data overrides UnitAvailability and updates the material*/
	UFUNCTION(BlueprintCallable,Category="UnitVolumeScene")
	void SetUnitVolumeSceneData(const FUnitData& InData);
	UFUNCTION(BlueprintCallable,Category="UnitVolumeScene")
	FUnitData GetUnitVolumeSceneData()const;

	UFUNCTION(BlueprintCallable,Category="UnitVolumeScene")
	void InitializedMaterialReference();
	UFUNCTION(BlueprintCallable,Category="UnitVolumeScene")
	void UpdateMeshMaterial() const;

#if WITH_EDITOR
	virtual void PostEditChangeChainProperty(struct FPropertyChangedChainEvent& PropertyChangedEvent) override;
	virtual void PostEditMove(bool bFinished) override;
	virtual void PostDuplicate(bool bDuplicateForPIE) override;
	virtual void PostLoad() override;
	virtual void PostActorCreated() override;
	void Ed_UpdateAssignationVolumeColor();
	
#endif

protected:
	
	// MATERIAL SECTION
	void CreateMaterialFromUnitVolumeConfig(const UUnitVolumeConfig* InUnitVolumeConfig);
	FLinearColor GetColorFromUnitState() const;
	
	virtual void BeginPlay() override;
	void UpdateUnitMesh();
	
	UPROPERTY(EditAnywhere) TObjectPtr<UStaticMeshComponent> UnitVolumeStaticMesh;
private:

	bool bUnitVolumeVisible;
	bool bSelectedUnitVolume;
	UPROPERTY() FUnitData UnitVolumeInfo;
	UPROPERTY() TObjectPtr<UMaterialInstanceDynamic> MaterialVolumeReference;
	UPROPERTY() TObjectPtr<UMaterialInstanceDynamic> MaterialVolumeFrameReference;
};