// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "UnitsFilterManager.h"
#include "UnitVolumeActor.h"
#include "IdealTwinCommon/ProjectRequestsManager/IdealTwinProjectRequestManager.h"
#include "UnitVolumesManager.generated.h"


class UUnitsFilterManager;
class UPostProcessComponent;
class UIdealTwinUnitResultHandler;
DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnUnitVolumeSelectionClear);
DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnUnitVolumeSelectionChange,const TArray<AUnitVolumeActor*>&, UnitVolumes);
DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnUnitDataRecive,const TArray<FUnitData>&, UnitData);


UENUM(BlueprintType) enum EUnitVolumeSelectionMode
{
	EHVSM_Single      UMETA(DisplayName = "Single"),
	EHVSM_Multi      UMETA(DisplayName = "Multi")
};


/** AUnitVolumesManager is a manager class that is in charge of managing the AUnitVolumeActor found in the level.

	Manage:
	  - Visibility of volumes
	  - Visibility of volumes by applying filters
	  - Selection/Deselection of volumes
	  - Assignment of the FUnitData based on the Unit volume Id
	
	It is prepared to receive when a UnitVolumeScene is selected and to centralize when the selection delegates.
	
	Note: right now you can only use the simple selection.
*/
UCLASS(HideCategories=(Collision,HLOD,Physics,Networking))
class IDEALTWINPRO_API AUnitVolumesManager : public AActor
{
	GENERATED_BODY()

public:
	AUnitVolumesManager(const FObjectInitializer& ObjectInitializer);
	
	UPROPERTY(VisibleAnywhere)
	TObjectPtr<UPostProcessComponent> PostProcessComponent;
	
	/* Called when selection changes*/
	UPROPERTY(BlueprintAssignable)
	FOnUnitVolumeSelectionChange OnUnitVolumeSelectionChange;
	/* Called when the selection has been cleared*/
	UPROPERTY(BlueprintAssignable)
	FOnUnitVolumeSelectionClear OnUnitVolumeSelectionClear;
	/* It is called when it has received the data from the database.*/
	UPROPERTY(BlueprintAssignable)
	FOnUnitDataRecive OnUnitDataReceive;


	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="UnitVolumeManager")
	bool bApplyFilterManagerChanges;
	
	/*	Selection mode:
		- Single: select only one volume. If a new one is selected, the old one selected becomes deselected
	 */
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="UnitVolumeManager")
	TEnumAsByte<EUnitVolumeSelectionMode> UnitVolumeSelectionMode; 

	/*  Returns the UnitVolumesManager reference of the scene. Returns null if there is none and returns the first
		if there is more than one */
	UFUNCTION(BlueprintCallable, Category="UnitVolumesManager", meta=(WorldContext="WorldContext"))
	static AUnitVolumesManager* GetUnitVolumesManager(const UObject* WorldContext);

	/* Clears the current list of references and does a scan of all UnitVolumeScenes in the level.*/
	UFUNCTION(BlueprintCallable, Category="UnitVolumesManager")
	void RefreshUnitVolumeScenesReferences();

	/**
	 * @brief Change UnitVolume visibility by Id list 
	 * @param IndexList Index list given to apply visibility
	 * @param bVisible Show/Hidde UnitVolume into the index list
	 * @param bResetAllVisibility
	 * @return if there were some UnitVolume changed return true.
	 */
	UFUNCTION(BlueprintCallable,BlueprintPure= false,Category="UnitVolumesManager")
	bool ApplyVisibilityByIndexList(const TArray<int32>& IndexList, bool bVisible, bool bResetAllVisibility= true);
	UFUNCTION(BlueprintCallable,BlueprintPure= false,Category="UnitVolumesManager")
	bool ApplyVisibilityByKeycodeList(const TArray<FString>& KeycodeList, bool bVisible, bool bResetAllVisibility= true);

	bool ApplyVisibilityToUnitVolumeList(const TArray<AUnitVolumeActor*>& UnitVolumeList, bool bVisible, bool bResetAllVisibility= true);
	
	UFUNCTION(BlueprintPure,Category="UnitVolumesManager|Selection")
	AUnitVolumeActor* GetSelectedUnitVolumeActor();

	UFUNCTION()
	void SelectUnitVolume(AUnitVolumeActor*UnitVolume, bool bSelection);

	/* Clears the UnitVolume selection. Call the OnUnitVolumeSelectionClear event*/
	UFUNCTION(BlueprintCallable, Category="UnitVolumesManager|Selection")
	void ClearUnitVolumeSelection();
	
	UFUNCTION(BlueprintPure, Category="UnitVolumesManager")
	TArray<AUnitVolumeActor*> GetUnitVolumesList() const;

	/* Get Unit list from Wrapper and assign them to UnitVolumeActors into the level*/
	void FillUnitDataAndAssign();
	/* Makes an assignment of the FUnitData to each of the UnitVolumeActors based on their VolumeId*/
	UFUNCTION(BlueprintCallable, Category="UnitVolumesManager")
	void AssignUnitDataToVolume(const TArray<FUnitData>& InUnitDataArray);

	/* Make a call to the database to obtain the data of the UnitGroup and the Units it contains*/
	UFUNCTION(BlueprintCallable,BlueprintNativeEvent, Category="UnitVolumesManager|Request")
	void InitializedUnitData();
	void InitializedUnitData_Implementation();
	

	/* Set all Unit Volumes Visibility to false*/
	UFUNCTION(BlueprintCallable, Category="UnitVolumesManager")
	void TurnOffUnitVolumesVisibility();
	UFUNCTION(BlueprintCallable, Category="UnitVolumesManager")
	void TurnOnUnitVolumesVisibility(bool bUseFIilters);
	
	UFUNCTION(BlueprintPure, Category="UnitVolumesManager")
	TArray<FUnitData> GetUnitDataFromManager()const { return UnitDataArray; }

	UFUNCTION(BlueprintPure, Category="UnitVolumesManager")
	UUnitsFilterManager* GetUnitFilterManager()const { return FilterManager; }
	
	UFUNCTION()
	void OnResponseUnitData(class UIdealTwinUnitResultHandler* ResultData, class UIdealTwinErrorHandler* Error);

	UFUNCTION(BlueprintPure,Category="UnitVolumesManager")
	TArray<AUnitVolumeActor*> GetVisibleVolumes()const;
	
	void SetVisibleVolumes(const TArray<AUnitVolumeActor*>& InVolumes);
protected:
	void InitializeFilterManager();

	
	/* Does a scan of all UnitVolumeScenes with RefreshUnitVolumeScenesReferences() and performs DoUnitsRequest()*/
	virtual void BeginPlay() override;

	UFUNCTION() void Received_ProjectManagerStateChanged(TEnumAsByte<EProjectManagerState> NewState);
	UFUNCTION()	void Received_FilterManagerResultChanged(const FUnitsFilterResult& FilterResult);
	UFUNCTION() void Received_UpdateData(EUpdateRequestDataType DataType, EUpdateRequestType RequestType);
	
private:
	UPROPERTY()	TObjectPtr<UUnitsFilterManager> FilterManager;

	UPROPERTY() TArray<FUnitData> UnitDataArray;
	UPROPERTY() TArray<AUnitVolumeActor*> UnitVolumeSelection;
	UPROPERTY() TArray<AUnitVolumeActor*> UnitVolumeScenes;
	UPROPERTY() TArray<AUnitVolumeActor*> UnitVolumesVisible;	
};
