﻿// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "UObject/Object.h"
#include "IdealTwinAPI/RealEstateAPI/Models/IdealTwinRealEstateDataModels.h"
#include "UnitsFilterManager.generated.h"


/*
Filter classes are a way of being able to read and compare properties of a structure from c++ or
blueprint.

The idea is do comparison with FUnitData data structure properties and filter UnitDataList array.

The properties are enumerated 4 different enumerators, 3 are the important ones:
- EUnitPropertyStringFilter: These are the grouped properties that have a string type. We can do are the comparisons by coincidence.
- EUnitPropertyNumberFilter: These are the grouped properties that have a number that could be float or int type. We can
do are range comparisons between a minimum and a maximum.
- EUnitPropertySpecificFilter: They are the grouped properties that have an Enumerator type. We can do are the
comparisons by coincidence.

These types of properties have a specific class to handle them (UUnitFilterString, UFilterNumber, UFilterSpecific)
and be added to the UUnitFilterCollection to execute comparison operation.
*/


class AUnitVolumesManager;
struct FUnitData;
class AUnitVolumeActor;
///////////////////////////////////////////////////////////////////////////////////
/// SPECIFIC STRUCTURES PROPERTY
UENUM(BlueprintType) enum EUnitPropertyFilter
{
	EHPF_Keycode				UMETA(Displayname="(str) Keycode"),
	EHPF_Availability			UMETA(Displayname="Availability"),
	EHPF_Displayname			UMETA(Displayname="(str) DisplayName"),	
	EHPF_Reference				UMETA(Displayname="(str) Reference"),
	EHPF_Price					UMETA(Displayname="(float) Price"),	
	EHPF_Rooms					UMETA(Displayname="(int) Rooms"),
	EHPF_Bathrooms				UMETA(Displayname="(int) Bathrooms"),
	EHPF_DateReservation		UMETA(Displayname="(str) Date Reservation"),
	EHPF_DateSold				UMETA(Displayname="(str) Date Sold"),
	EHPF_Phase					UMETA(Displayname="(str) Phase"),	
	EHPF_Area					UMETA(Displayname="(float) Area"),
	EHPF_AreaBuilt				UMETA(Displayname="(float) Area Built"),
	EHPF_AreaUseful				UMETA(Displayname="(float) Area UseFul"),
	EHPF_AreaCommon				UMETA(Displayname="(float) Area Common"),
	EHPF_AreaTerrace			UMETA(Displayname="(float) Area Terrace"),
	EHPF_ExternalId				UMETA(Displayname="(str) External ID"),
	EHPF_VolumeId				UMETA(Displayname="(int) VolumenId"),
	//ADDRESS PROPERTIES
		EHPF_Building			UMETA(Displayname="(str) Building"),
		EHPF_Staircase			UMETA(Displayname="(str) Staircase"),
		EHPF_Floor				UMETA(Displayname="(int) Floor"),
		EHPF_FloorLetter		UMETA(Displayname="(str) Floor Letter"),
		EHPF_Normalied			UMETA(Displayname="(str) Normalized"),
	EHPF_HomeType				UMETA(Displayname="Home Type"),
};
UENUM(BlueprintType) enum EUnitPropertyStringFilter
{
	EHPSF_Keycode			UMETA(Displayname="Keycode"),
	EHPSF_Displayname		UMETA(Displayname="DisplayName"),	
	EHPSF_Reference			UMETA(Displayname="Reference"),
	EHPSF_DateReservation	UMETA(Displayname="Date Reservation"),
	EHPSF_DateSold			UMETA(Displayname="Date Sold"),
	EHPSF_Phase				UMETA(Displayname="Phase"),
	EHPSF_ExternalId		UMETA(Displayname="External ID"),

	EHPSF_Building			UMETA(Displayname="Building"),
	EHPSF_Staircase			UMETA(Displayname="Staircase"),
	
	EHPSF_FloorLetter		UMETA(Displayname="FloorLetter"),
	EHPSF_Normalized		UMETA(Displayname="Normalized"),

};
UENUM(BlueprintType) enum EUnitPropertyNumberFilter
{
	EHPNF_Price				UMETA(Displayname="Price"),	
	EHPNF_Rooms				UMETA(Displayname="Rooms"),
	EHPNF_Bathrooms			UMETA(Displayname="Bathrooms"),
	EHPNF_Floor				UMETA(Displayname="Floor"),
	EHPNF_Area				UMETA(Displayname="Area"),
	EHPNF_AreaBuilt			UMETA(Displayname="Area Built"),
	EHPNF_AreaUseful		UMETA(Displayname="Area UseFul"),
	EHPNF_AreaCommon		UMETA(Displayname="Area Common"),
	EHPNF_AreaTerrace		UMETA(Displayname="Area Terrace"),
	EHPNF_VolumeId			UMETA(Displayname="VolumenId"),
	

};
UENUM(BlueprintType) enum EUnitPropertySpecificFilter
{
	EHPSpF_Availability		UMETA(DisplayName="Availability"),
	EHPSpF_HomeType			UMETA(DisplayName="UnitType"),
};

UENUM(BlueprintType) enum EUnitFilterPassType
{
	EHFPT_Explicid		UMETA(DisplayName="Explicid"),
	EHFPT_Exclude		UMETA(DisplayName="Exclude"),
	EHFPT_Include		UMETA(DisplayName="Include"),
};



///////////////////////////////////////////////////////////////////////////////////
/// FILTERS PROPERTY
UCLASS(BlueprintType,Transient)
class IDEALTWINPRO_API UUnitFilter : public UObject
{
	GENERATED_BODY()
public:
	DECLARE_DELEGATE(FOnFilterUpdated)
	FOnFilterUpdated OnFilterUpdated;
	
	UPROPERTY(EditDefaultsOnly,BlueprintReadWrite,Category="UnitFilter|Filters")
	TEnumAsByte<EUnitFilterPassType> FilterPassType = EHFPT_Explicid;
	
	virtual bool HasPassFilter(const FUnitData& DataCompare)const {return false;}
	virtual TArray<FUnitData> FilterUnitData(const TArray<FUnitData>& DataCompare)const {return {};}
	virtual TArray<FUnitData> ExcludeFromUnitData(const TArray<FUnitData>& DataCompare)const {return {};}
	UFUNCTION(BlueprintCallable,Category="UnitFilter|Filters")
	void UpdatedFilter(); 

	
	UPROPERTY(EditDefaultsOnly,BlueprintReadWrite,Category="UnitFilter|Filters")
	bool bApplyFilter = true;
};
///////////////////////////////////////////////////////////////////////////////////
/// CUSTOM PROPERTY
UCLASS(Abstract, Blueprintable,NonTransient)
class IDEALTWINPRO_API UUnitFilterCustom : public UUnitFilter
{
	GENERATED_BODY()
public:
	UFUNCTION(BlueprintPure, Category="UnitFilter|Filters")
	static UUnitFilterCustom* FilterCustomByClass(TSubclassOf<UUnitFilterCustom> InCustomFilterClass, TEnumAsByte<EUnitFilterPassType>
												InFilterPassType, bool bEnableFilter=true);

	
	UFUNCTION(BlueprintImplementableEvent,Category="UnitFilter|Filters")
	bool PassFilter(const FUnitData& InUnitData)const;

	
	virtual bool HasPassFilter(const FUnitData& DataCompare)const override;
};

///////////////////////////////////////////////////////////////////////////////////
/// STRING PROPERTY
UCLASS() class IDEALTWINPRO_API UUnitFilterString : public UUnitFilter
{
	GENERATED_BODY()
public:
	UFUNCTION(BlueprintPure, Category="UnitFilter|Filters")
	static UUnitFilterString* FilterMatchString(TEnumAsByte<EUnitPropertyStringFilter> InProperty, const FString& InString, TEnumAsByte<EUnitFilterPassType>
	                                            InFilterPassType, bool bEnableFilter=true);
	static EUnitPropertyFilter ConvertPropStringToPropFilter(EUnitPropertyStringFilter InFilter);
	virtual bool HasPassFilter(const FUnitData& DataCompare) const override;
	inline bool CompareString(const FString& Comp) const;
private:
	FName _PropName;
	EUnitPropertyStringFilter _Property;
	FString _String;
};


///////////////////////////////////////////////////////////////////////////////////
/// NUMBER PROPERTY
UCLASS(BlueprintType,NotBlueprintable,Transient)
class IDEALTWINPRO_API UFilterNumber : public UUnitFilter
{
	GENERATED_BODY()
public:
	UFUNCTION(BlueprintPure, Category="UnitFilter|Filters")
	static UFilterNumber* FilterNumberRange(TEnumAsByte<EUnitPropertyNumberFilter> InProperty, TEnumAsByte<EUnitFilterPassType> InFilterPassType, float InMin = -1, float InMax = 99999999, bool bEnableFilter = true);
	static EUnitPropertyFilter ConvertPropNumberToPropFilter(EUnitPropertyNumberFilter InFilter, bool& bFloat);
	inline bool IsValueInRange(float InValue) const;
	virtual bool HasPassFilter(const FUnitData& DataCompare) const override;
private:
	EUnitPropertyNumberFilter _Property;
	float _Min;
	float _Max;
};
///////////////////////////////////////////////////////////////////////////////////
/// SPECIFIC PROPERTY
UCLASS()
class IDEALTWINPRO_API UFilterSpecific : public UUnitFilter
{
	GENERATED_BODY()
public:
	UFUNCTION(BlueprintPure, Category="UnitFilter|Filters")
	static UFilterSpecific* FilterEnum(TEnumAsByte<EUnitPropertySpecificFilter> Enum, uint8 EnumByte, TEnumAsByte<EUnitFilterPassType> InFilterPassType, bool bEnableFilter = true);

	virtual bool HasPassFilter(const FUnitData& DataCompare) const override;
	virtual TArray<FUnitData> FilterUnitData(const TArray<FUnitData>& DataCompare) const override;
	virtual TArray<FUnitData> ExcludeFromUnitData(const TArray<FUnitData>& DataCompare) const override;
private:
	EUnitPropertySpecificFilter _Property;

	EAvailabilityState _Availability;
	EUnitBuildingType _HomeType;
};


/** Represents a collection of filters that can be applied to unit data. The class
 * enables creating, managing, and applying filters as well as triggering related events.
 */
UCLASS(BlueprintType,Transient) class IDEALTWINPRO_API UUnitFilterCollection : public UObject
{
	GENERATED_BODY()
public:
	UUnitFilterCollection(const FObjectInitializer& ObjectInitializer);
	
	DECLARE_DELEGATE(FOnFilterCollectionChanged)
	FOnFilterCollectionChanged OnFilterCollectionChanged;
	TEnumAsByte<EUnitFilterPassType> FilterCollectionType;

	/** Creates and initializes a UUnitFilterCollection instance with the given filters and initialization flag.
	 *
	 * @param InFilters The array of UUnitFilter objects used to initialize the filter collection.
	 * @param bInitialization A boolean indicating whether the collection is in the initialization phase.
	 * @return A pointer to the created and configured UUnitFilterCollection object. */
	UFUNCTION(BlueprintCallable, Category="UnitFilterManager|UnitFilterCollection",meta=(AutoCreateRefTerm="InFilters"))
	static UUnitFilterCollection* CreateUnitFilterCollection(TArray<UUnitFilter*> InFilters, bool bInitialization);
	UFUNCTION(BlueprintPure, Category="UnitFilterManager|UnitFilterCollection")
	bool IsOnInitialization()const;
	UFUNCTION(BlueprintCallable, Category="UnitFilterManager|UnitFilterCollection")
	void FinishInitialization();

	UFUNCTION(BlueprintCallable, Category="UnitFilterManager|UnitFilterCollection")
	void AddFilter(UUnitFilter* NewFilter);
	UFUNCTION(BlueprintCallable, Category="UnitFilterManager|UnitFilterCollection")
	void ClearFilterCollection();
	UFUNCTION(BlueprintCallable, Category="UnitFilterManager|UnitFilterCollection")
	void ReplaceFilters(TArray<UUnitFilter*> InFilters);
	UFUNCTION(BlueprintCallable, Category="UnitFilterManager|UnitFilterCollection")
	void FilterCollectionUpdated();
	
	TArray<UUnitFilter*> GetFilters()const;
	TArray<FUnitData> FilterUnitDataWithFilters(const TArray<FUnitData>& InUnitData) const;
private:
	UPROPERTY() TArray<UUnitFilter*> FiltersList;
	/* When set to true, this flag prevents any changes to the filters in the manager during the components' initialization phase. This ensures that the filtering system 
	remains stable and avoids unintended modifications or inconsistent behavior while components are being initialized.*/
	bool bFilterInitialization;
};


///////////////////////////////////////////////////////////////////////////////////
/// UTILITIES PROPERTY
UCLASS(NotPlaceable,NotBlueprintable)
class IDEALTWINPRO_API UUnitFilterUtils : public UObject
{
	GENERATED_BODY()
public:
	static FString GetUnitPropertyParse(EUnitPropertyFilter PropertyFilter);
	static float GetUnitPropertyFloat(const FUnitData& UnitData, EUnitPropertyFilter PropertyFilter, bool bFloatNumber);
	static int32 GetUnitPropertyInt(const FUnitData& UnitData, EUnitPropertyFilter PropertyFilter);
	static FString GetUnitPropertyString(const FUnitData& UnitData, EUnitPropertyFilter PropertyFilter);
};

/* Template to extract the value of FUnitData property property's name*/
template<typename DataType>
static DataType GetDataInUnitData(const FUnitData& UnitData, const FString& PropertyParse)
{
	FString PropName;
	FString SubPropName;
	if(!PropertyParse.Split(".",&PropName,&SubPropName))
		PropName = PropertyParse;
	
	FProperty* Property = UnitData.StaticStruct()->FindPropertyByName(FName(PropName));
	if (const FStructProperty* PropStruct = CastField<FStructProperty>(Property))
	{
		void const* SubPropAccess = Property->ContainerPtrToValuePtr<void>(&UnitData);
		FProperty* SubProp = PropStruct->Struct->FindPropertyByName(FName(SubPropName));
		return *SubProp->ContainerPtrToValuePtr<DataType>(SubPropAccess);
	}
	return *Property->ContainerPtrToValuePtr<DataType>(&UnitData);
}

USTRUCT(Blueprintable) struct FUnitsFilterResult
{
	GENERATED_BODY()
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="UnitsFilterApplyResult") int32 Selection;
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="UnitsFilterApplyResult") int32 Total;
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="UnitsFilterApplyResult") TArray<AUnitVolumeActor*> FilteredUnitVolumes;
	
	FUnitsFilterResult(): Selection(0),Total(0){}
	FUnitsFilterResult(int32 InSelection, int32 InTotal, const TArray<AUnitVolumeActor*>& InFilteredUnitVolumes = {})
		:Selection(InSelection), Total(InTotal),FilteredUnitVolumes(InFilteredUnitVolumes){}
};

/*	UnitsFilterManager is a class to be created inside event graph and be used as manager to do comparative filters and
	be applied to Unit Volumes into World */
UCLASS(BlueprintType)
class IDEALTWINPRO_API UUnitsFilterManager : public UObject
{
	GENERATED_BODY()
public:
	UUnitsFilterManager(const FObjectInitializer& ObjectInitializer);

	DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnClearAllFilters);
	UPROPERTY(BlueprintAssignable, Category="UnitsFilterManager")
	FOnClearAllFilters OnClearAllFilters;
	DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnFilterResultChanged, const FUnitsFilterResult&, FilterResult);
	UPROPERTY(BlueprintAssignable, Category="UnitsFilterManager")
	FOnFilterResultChanged OnFilterResultChanged;
	
	/* Create UnitsFilterManager instance to handle filters*/
	UFUNCTION(BlueprintCallable, Category="UnitsFilterManager")
	static UUnitsFilterManager* CreateUnitsFilterManager();
	/* Apply filter in FUnitData array and set UnitVolumeScene visible or hidden*/
	UFUNCTION(BlueprintCallable, Category="UnitsFilterManager")
	static FUnitsFilterResult ApplyFiltersToUnitVolume(UPARAM(ref)TArray<FUnitData> Data,
	                                                   TArray<AUnitVolumeActor*> UnitVolumeScene,
	                                                   UPARAM(ref)TArray<UUnitFilter*> Filters);
	UFUNCTION(BlueprintCallable, Category="UnitsFilterManager")
	static TArray<FUnitData> FilterUnitDataWithFilters(const TArray<FUnitData>& InUnitData,
	                                                   const TArray<UUnitFilter*> InFilters);
	

	UFUNCTION(BlueprintCallable, Category="UnitsFilterManager")
	static TArray<AUnitVolumeActor*> FilteredUnitVolumesFromUnitData(const TArray<FUnitData>& InUnitData,
	                                                                 TArray<AUnitVolumeActor*> InUnitVolumeActors);
	UFUNCTION(BlueprintCallable, Category="UnitsFilterManager")
	static FUnitsFilterResult ApplyFiltersToUnitVolumeManager(AUnitVolumesManager* UnitVolumeManager,UPARAM(ref)TArray<UUnitFilter*> Filters);
	
	
	UFUNCTION(BlueprintCallable, Category="UnitsFilterManager")
	void SetUnitData(const TArray<FUnitData>& InUnitData);
	UFUNCTION(BlueprintCallable, Category="UnitsFilterManager")
	void SetUnitVolumeScene(const TArray<AUnitVolumeActor*>& InUnitVolumeScenes);
	UFUNCTION(BlueprintCallable, Category="UnitsFilterManager")
	void ClearAllFilters();
	
	
	// FILTER COLLECTION
	UFUNCTION(BlueprintCallable,Category="UnitsFilterManager")
	void RegisterFilterCollection(UUnitFilterCollection* InFilterCollection);
	UFUNCTION(BlueprintCallable,Category="UnitsFilterManager")
	void UnregisterFilterCollection(UUnitFilterCollection* InFilterCollection, bool bApplyFilterAfterUnregister);
	UFUNCTION(BlueprintCallable,Category="UnitsFilterManager")
	FUnitsFilterResult GetCurrentFilterResult();
	UFUNCTION(BlueprintCallable,Category="UnitsFilterManager")
	FUnitsFilterResult ApplyFilterCollections();

	UFUNCTION(BlueprintCallable,Category="UnitsFilterManager")
	void ApplyFilterResultToUnitActors();
	
	
 	UFUNCTION(BlueprintCallable, Category="UnitsFilterManager| Helpers")
	static bool GetUnitDataNumberRange(const TArray<FUnitData>& InUnitData, TEnumAsByte<EUnitPropertyNumberFilter> Property, float& OutMin, float& OutMax);
	UFUNCTION(BlueprintCallable, Category="UnitsFilterManager| Helpers")
	static bool GetUnitDataStringsOptions(const TArray<FUnitData>& InUnitData, TEnumAsByte<EUnitPropertyStringFilter> Property, TArray<FString>& OutElements);

protected:
	void Received_FilterCollectionChanged();
	
private:
	UPROPERTY() TArray<UUnitFilterCollection*> FiltersCollection;

	UPROPERTY() FUnitsFilterResult CurrentFilterResult;
	UPROPERTY() TArray<FUnitData> NoFilteredUnitData;
	UPROPERTY() TArray<AUnitVolumeActor*> UnitVolumeScenes;
};
