﻿#pragma once 
#include "CoreMinimal.h"
#include "GameplayTagContainer.h"
#include "GameFramework/Character.h"
#include "GameFramework/SwitcherPawn/PawnSwitchInterface.h"
#include "IdealTwinBaseCharacter.generated.h"


class UIdealTwinViewConfig;
class UIdealTwinInputComponent;
class UInputAction;
struct FInputActionValue;
class USpringArmComponent;
class UInputMappingContext;
class UCameraComponent;
class USphereComponent;

DECLARE_DYNAMIC_MULTICAST_DELEGATE_ThreeParams(FOnPlayerTouchMoveEvent, float, DeltaMoveLength, FVector2D, DeltaMove, FVector2D, TouchPosition);
DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnPlayerTouchEvent, FVector2D, TouchPosition);


/*	Abstract base class to create different character with specifics implementation o easy to setup.
	This class implements:
		- Camera smooth movement and rotation using SpringArmComponent
		- Auto Registration for InputMappingContext
		- InitialZoom with min/max values and smooth transition
		- Rotation view clamping to restrict camera angles
		- Touch Support
		- Possessed/UnPossessed include some process to optimized input registration and tick
 */
UCLASS(Abstract)
class IDEALTWINPRO_API AIdealTwinBaseCharacter : public ACharacter, public IPawnSwitchInterface
{
	GENERATED_BODY()

public:
	
	// Sets default values for this character's properties
	AIdealTwinBaseCharacter(const FObjectInitializer& ObjectInitializer);

	///////////////////////////////////////////
	/// BASE COMPONENTS
	//////////////////////////////////////////


	// IPawnSwitchInterface Begin
	virtual FGameplayTag GetSwitchPawnTag_Implementation() const override;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinCharacter|SwitchPawn")
	bool bAutoRegisterPawnToSwitcher;
	/* Tag to be used by SwitcherPawnController Component*/
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinCharacter|SwitchPawn")
	FGameplayTag SwitchPawnTag;
	// IPawnSwitchInterface End
	
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Instanced,Category="IdealTwinCharacter")
	TObjectPtr<UIdealTwinViewConfig> ViewConfig;
	
	
	UPROPERTY(VisibleAnywhere,BlueprintReadOnly,Category= "IdealTwinCharacter")
	TObjectPtr<USpringArmComponent> SpringArmObject;
	UPROPERTY(VisibleAnywhere,BlueprintReadOnly,Category= "IdealTwinCharacter")
	TObjectPtr<UCameraComponent> CameraCharObject;
	UPROPERTY(VisibleAnywhere,BlueprintReadOnly,Category= "IdealTwinCharacter")
	TObjectPtr<USphereComponent> SphereCameraCollision;
	
	UPROPERTY(VisibleAnywhere ,BlueprintReadOnly, Category="IdealTwinCharacter|Input")
	TObjectPtr<UIdealTwinInputComponent> IdealTwinInputComponent;
	
	
	///////////////////////////////////////////
	/// CHARACTER MOVEMENT 
	//////////////////////////////////////////
	/*This multiplier is applied to the movement value to define the velocity.
	 It affects the MaxWalkSpeed parameter of the UCharacterComponent, which limits the movement speed.
	 The value range is normalized from 0.f to 2.f, where 1.f represents 0.5f of the MaxWalkSpeed.*/
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "IdealTwinCharacter|Movement", meta=(ClampMin="0", UIMin = "0", UIMax = "2"))
	float MovementSpeedMultiplier;
	
	///////////////////////////////////////////
	/// ZOOM UTILITIES
	///////////////////////////////////////////
	
	
	


	/** Controls how aggressively mouse motion translates to character rotation in the pitch axis. */
	UPROPERTY(EditAnywhere, BlueprintReadWrite,DisplayName="VerticalCameraMultiplier", Category = "IdealTwinCharacter|ViewCamera|CameraSpeedMultiplier", meta=(UIMin = "0", UIMax = "1"))
	float MouseSensitivityScale_Pitch;
	/** Controls how aggressively mouse motion translates to character rotation in the yaw axis. */
	UPROPERTY(EditAnywhere, BlueprintReadWrite,DisplayName="HorizontalCameraMultiplier", Category = "IdealTwinCharacter|ViewCamera|CameraSpeedMultiplier", meta=(UIMin = "0", UIMax = "1"))
	float MouseSensitivityScale_Yaw;
	
	
	// TOUCH INPUT
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "IdealTwinCharacter|TouchInput")
	FVector2D TouchLookSensibility;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "IdealTwinCharacter|TouchInput")
	bool bTouchLookInversion;
	
	virtual void SetupPlayerInputComponent(UInputComponent* PlayerInputComponent) override;
	
	virtual void Tick(float DeltaTime) override;
	
	//// INPUTS IMPLEMENTATION
	/*	Inputs Actions (Move,Look and InitialZoom) have a internal implementation but it could get the information on Blueprints Events
		to be able to extend the implementation.
	 */
	/* Triggered when character received inputs to Move*/
	UFUNCTION(BlueprintImplementableEvent, BlueprintCallable,Category= "IdealTwinCharater|Movement")
	void MoveCharacter(FVector2D MoveValue);
	virtual void MoveCharacter_Implementation(FVector2D MoveValue);
	/* Triggered when character received inputs to Look*/
	UFUNCTION(BlueprintImplementableEvent, BlueprintCallable,Category= "IdealTwinCharater|Movement")
	void LookCharacter(FVector2D LookValue);
	virtual void LookCharacter_Implementation(FVector2D LookValue);
	/* Triggered when character received inputs to InitialZoom*/
	UFUNCTION(BlueprintImplementableEvent, BlueprintCallable,Category= "IdealTwinCharater|Movement")
	void Zooming(float ZoomValue);
	UFUNCTION(BlueprintCallable, Category = "IdealTwinCharacter|Movement")
	virtual void Zooming_Implementation(float ZoomValue);
	UFUNCTION(BlueprintCallable, Category= "IdealTwinCharater|Movement")
	void SetZoomValue(float InValue);
	
	
	UFUNCTION(BlueprintCallable,Category= "IdealTwinCharater|Movement")
	void DoMoveCharacter(FVector2D MoveDelta);
	UFUNCTION(BlueprintCallable,Category= "IdealTwinCharater|Movement")
	void DoLookCharacter(FVector2D LookDelta, FVector2D Sensibility, FVector2D MaxSpeed = FVector2D(-1,-1));
	UFUNCTION(BlueprintCallable,Category= "IdealTwinCharater|Movement")
	void DoZoomCharacter(float ZoomDelta);
	
	/* Active smooth camera movement and rotation*/
	UFUNCTION(BlueprintCallable, Category = "IdealTwinCharacter|ViewCamera")
	void ActiveSmoothTrackCamera(bool bActivate);

	UFUNCTION(BlueprintImplementableEvent, BlueprintCallable,Category= "IdealTwinCharater|Movement")
	void TouchTab(FVector2D TouchPosition);
	UFUNCTION()
	virtual void TouchTab_Implementation(FVector2D TouchPosition);
	UFUNCTION(BlueprintImplementableEvent, BlueprintCallable,Category= "IdealTwinCharater|Movement")
	void TouchOneMove(float DeltaMoveLength ,FVector2D DeltaMove ,FVector2D TouchPosition);
	UFUNCTION()
	virtual void TouchOneMove_Implementation(float DeltaMoveLength ,FVector2D DeltaMove ,FVector2D TouchPosition);
	UFUNCTION(BlueprintImplementableEvent, BlueprintCallable,Category= "IdealTwinCharater|Movement")
	void TouchPinch(float DeltaMove, FVector2D CenterPosition);
	virtual void TouchPinch_Implementation(float DeltaMove, FVector2D CenterPosition);


	UFUNCTION(BlueprintCallable, Category = "IdealTwinCharacter|View")
	void SetViewConfig(UIdealTwinViewConfig* InViewConfig);
	UFUNCTION(BlueprintCallable, Category = "IdealTwinCharacter|View")
	void ResetViewConfig();
	void ApplyCurrentViewConfig() const;

	
	
	//TOUCH INPUT METHODS
protected:
	
	//// INPUTS RECEIVED
	UFUNCTION()void MoveCharacter_Received(FVector2D MoveValue);
	UFUNCTION()void LookCharacter_Received(FVector2D LookValue);
	UFUNCTION()void Zoom_Received(float ZoomValue);
	UFUNCTION()void TouchOneMove_Received(float DeltaMoveLength ,FVector2D DeltaMove ,FVector2D TouchPosition);
	UFUNCTION()void TouchTab_Received(FVector2D TouchPosition);

	

	/* Initialize camera rotation to a specific angle base on star camera angle*/
	virtual void BeginPlay() override;

	/* When the character is possessed active some process:
	 	- Register Input Mapping Context
		- Active Smooth Track camera if is set */
	virtual void PossessedBy(AController* NewController) override;
	/* Such as character is possessed, on UnPossessed disable:
		- Unregister Input Mapping Context 
		- Disable smooth track camera  */
	virtual void UnPossessed() override;

	/* Revert camera smooth parameters, this helps to stop camera updates when is unpossessed*/
	virtual void ResetCameraParams();

	/* Use clamp parameters to use view clamps by CameraManager*/
	UFUNCTION(BlueprintCallable, Category = "IdealTwinCharacter|ViewCamera")
	void ClampCameraRotation(bool bClamp);
	/* Load camera smooth camera parameters when is possessed*/
	void LoadSpringArmParams() const;

#if WITH_EDITOR
	virtual void PostEditMove(bool bFinished) override;
	virtual void PostEditChangeProperty(FPropertyChangedEvent& PropertyChangedEvent) override;
#endif

	FTimerHandle UpdateControllerRotatorDelayHandle;
	FTimerHandle ActivationSmoothDelayHandle;

	//TOUCH INPUT LOGIC
	FTimerHandle TapTimer;
	bool bMovingTouch = false;
	FVector2D TouchStartPos;
	FVector2D TouchLastPos;
	UPROPERTY(BlueprintReadWrite)
	bool bUsingTouch;

	UPROPERTY() const UInputAction* PinchInputAction;
	
	bool bPinchGesture;
	float LastPinchVal;

private:
	TObjectPtr<UIdealTwinViewConfig> Current_ViewConfig;
	
	UPROPERTY(BlueprintAssignable, Category = "IdealTwinCharacter|Touch", meta = (AllowPrivateAccess = true))
	FOnPlayerTouchMoveEvent OnPlayerTouchMoveEvent;
	UPROPERTY(BlueprintAssignable, Category = "IdealTwinCharacter|Touch", meta = (AllowPrivateAccess = true))
	FOnPlayerTouchEvent OnPlayerTouchEvent;
	
};











