﻿// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "GameplayTagContainer.h"
//#include "InputMappingContext.h"
#include "IdealTwinInputComponent.generated.h"

class UInputMappingContext;
class UEnhancedInputComponent;


/// INTERACTION DELEGATES
DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnInteractionPrimary);
DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnInteractionSecondary);
DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnInteractionTick,FVector2D, ScreenPosition);
/// DESKTOP DELEGATES
DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnMoveAction,FVector2D, MoveValue);
DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnLookAction,FVector2D, LookValue);
DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnZoomAction,	float, ZoomValue);
/// TOUCH DELEGATES
DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnTouchTab, FVector2D, TouchPosition);
DECLARE_DYNAMIC_MULTICAST_DELEGATE_ThreeParams(FOnOneTouchMove,float, DeltaMoveLength ,FVector2D, DeltaMove ,FVector2D, TouchPosition);
DECLARE_DYNAMIC_MULTICAST_DELEGATE_TwoParams(FOnTouchPinch,float, DeltaMove, FVector2D, CenterPosition);

DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnInputInactive);
DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnInputActive);

UENUM() enum EITWInputMode
{
	EIM_Desktop		UMETA(Displayname = "Desktop"),
	EIM_Touch		UMETA(Displayname = "Touch"),
};


/** UIdealTwinInputComponent is a class intended to have user input set and return concrete events of interaction to be
 	easily implemented on Pawns or PlayerController.
  	This class also registers the UInputContextMapping and thus have the necessary inputs to obtain in implementations.
  	like Move,Look or Zooming...etc.
  	
  	Implementations:
  	- CharacterMovement (Move, Look or Zooming)
  	- Touch Support (Tab, TouchMove, Pinch)
  	- Interaction interface.
  	
  	Additionally, it has an implementation for Touch Screens in which it will return desktop inputs on the click or the touch when
  	use touch screens.
  	
  	Interaction Mapping is included to detect the inputs as click to make interactions.
 */
UCLASS(BlueprintType,ClassGroup=(IdealTwin),HideCategories=(Variable,Navigation,Sockets,ComponentReplication,Replication,Collision,AssetUserData,Cooking,ComponentTick), meta=(BlueprintSpawnableComponent))
class IDEALTWINPRO_API UIdealTwinInputComponent : public UActorComponent
{
	GENERATED_BODY()
public:
	UIdealTwinInputComponent(const FObjectInitializer& ObjectInitializer);

	/*Tag to find this specific InputComponent if you want to bind to the events*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs")
	FGameplayTag InputComponentId;
	/*Block inputs in BeginPlay. Blocking the inputs is useful to block them until the pawn is possessed.*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs")
	bool bStartBlockInputs;
	/*If bAutoInitialize is true will register the InputContextMapping.If it's false it has to be InitializedInputComponent() for registration*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs")
	bool bAutoInitialize;

	/*Initialization of the component by registering the InputContextMapping and the bindings to methods*/
	UFUNCTION(BlueprintCallable,Category="IdealTwinInputs")
	void InitializedInputComponent();
	
	/*Registers the InputContextMapping specified with bRegisterInteractionInputs,bRegisterTouchInputs,bRegisterDesktopInputs*/
	UFUNCTION(BlueprintCallable,Category="IdealTwinInputs|DesktopInput")
	bool RegisterInputsMapping();
	/*Unregister the previously registered InputContextMapping*/
	UFUNCTION(BlueprintCallable,Category="IdealTwinInputs|DesktopInput")
	void UnRegisterInputsMapping();
	
	/*Sets the state of BlockComponentInputs. If it's true the inputs will be blocked*/
	UFUNCTION(BlueprintCallable,Category="IdealTwinInputs")
	void BlockComponentInputs(bool bBlock);

	/** Looks for and returns a UIdealTwinInputComponent with a specific tag.
	 * @return Returns false if no component with the specified tag has been found*/
	UFUNCTION(BlueprintCallable, Category="IdealTwinInputs",meta=(DefaultToSelf, WorldContext="WorldContext"))
	static bool GetIdealTwinInputByTag(UObject* WorldContext,FGameplayTag ComponentTag, UIdealTwinInputComponent*& OutComponent);

	virtual void BeginPlay() override;
	virtual void TickComponent(float DeltaTime, ELevelTick TickType,
		FActorComponentTickFunction* ThisTickFunction) override;
	
protected:
	/*InputComponent to do the bindings*/
	UPROPERTY() UEnhancedInputComponent* EnhancedInputComponent;

	/*Desktop or touch detection status*/ 
	EITWInputMode CurrentInputMode;
	/*Changes the state of the InputMode between Touch and Desktop if click or touch has been pressed*/
	void ChangeDetectionMode(const FInputActionValue& InputValue, EITWInputMode Mode);

	/*	Binding of actions within the Input Context Mapping.
		Note: the bindings are done with FName manually with the names of the actions at IdealTwinPro/Core/Input/.
		If these UActions change names it will cause problems with the internal handling of event detection.*/
	void BindInputsActions();
	
	/*Block state of the inputs in the component. If this is true the inputs will not execute the events*/
	bool bBlockInputs;
private:
	bool bBlockedInputsStart = false;
	
	/// MOVEMENTS INPUTS
	//////////////////////////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////////////////////////
public:

	/*Register the UInputMappingContext corresponding to the Movement inputs*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs|Movement")
	bool bRegisterMovementInputs;
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs|Movement")
	TObjectPtr<UInputMappingContext> MovementInputMappingContext;

	/*Event that is triggered by Motion inputs*/
	UPROPERTY(BlueprintAssignable,Category="IdealTwinInputs|Movement")
	FOnMoveAction OnMoveAction;
	/*Event that is triggered by Look inputs*/
	UPROPERTY(BlueprintAssignable,Category="IdealTwinInputs|Movement")
	FOnLookAction OnLookAction;
	/*Event that is triggered by Zooming inputs*/
	UPROPERTY(BlueprintAssignable,Category="IdealTwinInputs|Movement")
	FOnZoomAction OnZoomAction;
	//Delegate that will trigger once the input gets into inactive state
	UPROPERTY(BlueprintAssignable,Category="IdealTwinInputs|Activation")
	FOnInputInactive OnInputInactive;
	//Delegate that will trigger once the input gets into Active state
	UPROPERTY(BlueprintAssignable,Category="IdealTwinInputs|Activation")
	FOnInputActive OnInputActive;

protected:
	UFUNCTION()void MoveAction_Received(const FInputActionValue& InputValue);
	UFUNCTION()void LookAction_Received(const FInputActionValue& InputValue);
	UFUNCTION()void ZoomAction_Received(const FInputActionValue& InputValue);
	
	/// INTERACTION INPUTS
	//////////////////////////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////////////////////////
public:

	/*Register the UInputMappingContext corresponding to the Interaction inputs*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs|Interaction")
	bool bRegisterInteractionInputs;
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs|Interaction")
	TObjectPtr<UInputMappingContext> InteractionInputMappingContext;
	/*Activate the detection tick for interact elements to implement the Hover/Unhover functionality on the interaction actors.*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs|Interaction")
	bool bUseInteractionTick;
	/*Tick interval for the interaction*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs|Interaction", meta=(EditCondition="bUseInteractionTick"))
	float InteractionTickInterval;

	/*Event that fires when the interaction has been clicked*/
	UPROPERTY(BlueprintAssignable ,Category="IdealTwinInputs|Interacion")
	FOnInteractionPrimary OnInteractionPrimary;
	/*Event that fires when the secondary interaction has been clicked*/
	UPROPERTY(BlueprintAssignable, Category="IdealTwinInputs|Interacion")
	FOnInteractionSecondary OnInteractionSecondary;
	/*Event that is activated if the bUseInteractionTick is true. Ticks every in InteractionTickInterval time*/
	UPROPERTY(BlueprintAssignable, Category="IdealTwinInputs|Interacion")
	FOnInteractionTick OnInteractionTick;

protected:
	UFUNCTION() void PrimaryInteraction_Received(const FInputActionValue& InputValue);
	UFUNCTION() void SecondaryInteraction_Received(const FInputActionValue& InputValue);
	UFUNCTION() void TickInteraction_Received();

private:
	FTimerHandle InteractTickHandler;
	
	/// TOUCH INPUTS
	//////////////////////////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////////////////////////
public:
	/*Register the UInputMappingContext corresponding to the Touch inputs*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs|TouchInput")
	bool bRegisterTouchInputs;
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs|TouchInput")
	TObjectPtr<UInputMappingContext> TouchInputMappingContext;

	/*Event that fires when touch is detected as tab*/
	UPROPERTY(BlueprintAssignable,Category="IdealTwinInputs|TouchInput")
	FOnTouchTab OnTouchTab;
	/*Event that fires when touch finger is moving*/
	UPROPERTY(BlueprintAssignable,Category="IdealTwinInputs|TouchInput")
	FOnOneTouchMove OnOneTouchMove;
	/*Event that fires when use 2 fingers and detect pinch gesture*/
	UPROPERTY(BlueprintAssignable,Category="IdealTwinInputs|TouchInput")
	FOnTouchPinch OnTouchPinch;

	/*If true when the Tab is detected it will also trigger the OnInteractionPrimary event*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs|TouchInput")
	bool bUseTouchTabAsInteraction;
	/*Detect if the touch has moved before tabbing to cancel the tab timer and start the touch movement*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs|TouchInput|CheckFastMove")
	bool bUseCheckFastMove;
	/*Length touch to detect the move*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs|TouchInput|CheckFastMove", meta=(EditCondition="bUseCheckFastMove"))
	float CheckFastMoveLength;
	/*Delay to detect the Tab touch. If the press and release of the touch has been less than the set time, it will
	  be understood that a tab has been made and the event will be trigger.*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs|TouchInput")
	float TouchTabDetection;
	/*Threshold when touch is moving to discard invalid input values*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinInputs|TouchInput")
	float TouchThreshold;

	/*Return last Tab position*/
	UFUNCTION(BlueprintCallable, Category="IdealTwinInputs|TouchInput")
	bool GetTabPos(FVector2D& OutTabPos)const;
	/*Updates touch detection for move or pinch by the TouchIndex*/
	UFUNCTION(BlueprintCallable, Category="IdealTwinInputs|TouchInput")
	void UpdateTouchGesture(float Delta);



	

	//Tracks the time on when is the user going to be considered inactive
	UFUNCTION(BlueprintCallable)
	void TrackInactiveTimes();

protected:


	//Tracks if the player has been active in the game
	bool IsInputActive;
	//Amount of seconds to consider that the input is inactive (no input has been pressed)
	UPROPERTY(EditAnywhere, Category = "IdealTwinInputs|InactiveInput")
	float TimeForConsideringInactive;
	
private:
	//Tracks the time where the input is going to be considered inactive
	float CurrentInactiveInputTime;
	//Tracks the last time when the user pressed an input
	float LastInputTime;
	//Will set IsInputActive if the inactive time has passed
	void SetIfInputIsInactive();

#if WITH_EDITORONLY_DATA
	UPROPERTY(EditAnywhere, Category = "Input", meta = (AllowPrivateAccess = "true"))
	bool bShowInactiveInputTimers; 
#endif
	///// END REVIEW CODE
	//////////////////////////////////////////////////////////////////////////

	
protected: 
	
	UPROPERTY() FTimerHandle TabTimer;
	bool bTouchPressed = false;
	bool bDetectFastMove= false;
	
	/*Return touch position by index. If touch isn't pressed will return (0,0)*/
	FVector2D GetTouchPosition(ETouchIndex::Type TouchIndex) const;

	
	void TabTouch(FVector2D TouchPos);
	void OneFingerTouch();
	void PinchTouch();
	UFUNCTION() void TabTouchAbort();

	
	/*Check for fast touch movement*/
	void CheckFastMove();
	bool IsEnabledCheckFastMove()const;

	/*Checks if value is valid been out of threshold*/
	bool IsOutThreshold(float Value) const;
	
	UFUNCTION()void Received_Touch1(const FInputActionValue& InputValue, bool bActivated);
	UFUNCTION()void Received_Touch2(const FInputActionValue& InputValue, bool bActivated);
	UFUNCTION()void Received_Touch3(const FInputActionValue& InputValue, bool bActivated);
	UFUNCTION()void Received_Touch4(const FInputActionValue& InputValue, bool bActivated);
private:
	ETouchIndex::Type CurrentTouchIndex;
	TArray<FVector2D> TouchLastPositions;

	float LastPinchDistance;
	FVector2D TabLocation;
};



