﻿// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "IdealTwinBaseCharacter.h"
#include "IdealTwinCommon/IdealTwinUtilsCommon.h"
#include "IdealTwinPanoramicCharacter.generated.h"


class AIdealTwinPanoramicCharacter;

UENUM(BlueprintType) enum EITwPanoramicType
{
	EPT_180View		UMETA(DisplayName = "180 View"),
	EPT_360View		UMETA(DisplayName = "360 View"),
};

USTRUCT(BlueprintType) struct FPanoramicViewSetup
{
	GENERATED_BODY()
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="FPanoramicViewSetup") TObjectPtr<UStaticMesh> DisplayMesh;
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="FPanoramicViewSetup") TObjectPtr<UMaterial> DisplayMaterial;
	FPanoramicViewSetup(){}
};

/** Character for 360 or 180 panoramic view setting a texture 2d as a Dome or Screen */
DECLARE_DYNAMIC_DELEGATE_OneParam(FOnSphereSetUpCompleted,  bool, bSetUpSuccessful);
UCLASS(Experimental)
class IDEALTWINPRO_API AIdealTwinPanoramicCharacter : public AIdealTwinBaseCharacter
{
	GENERATED_BODY()
public:
	// Sets default values for this character's properties
	AIdealTwinPanoramicCharacter(const FObjectInitializer& ObjectInitializer);


	UPROPERTY(VisibleAnywhere,BlueprintReadOnly,Category= "IdealTwinPanoramicChar")
	TObjectPtr<UStaticMeshComponent> DisplayViewMesh;


	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category= "IdealTwinPanoramicChar")
	TEnumAsByte<EITwPanoramicType> ViewType;

	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category= "IdealTwinPanoramicChar")
	TMap<TEnumAsByte<EITwPanoramicType>,FPanoramicViewSetup> SetupByView;
	
	virtual void SetupPlayerInputComponent(UInputComponent* PlayerInputComponent) override;
	
	/* Triggered when character received inputs to Look*/
	UFUNCTION(BlueprintImplementableEvent, BlueprintCallable,Category= "IdealTwinPanoramicChar|Movement")
	void Look360Character(FVector2D LookValue);
	virtual void Look360Character_Implementation(FVector2D LookValue);

	/**
	 * Download the provided image and set it into the sphere material, once is completed lauch the OnSphereSetUpCompletedDelegate with the success state
	 * @param PanoramicChar - Character which sphere will be init
	 * @param ImageData - Ideal Twin Image data to be downloaded
	 * @param SizeImage - Size used when requesting the image
	 * @param OnSphereSetUpCompletedDelegate - Delegate launched when the process is completed
	 */
	UFUNCTION(BlueprintCallable, Category= "IdealTwinCharater|IdealTwinPanoramicChar|Render")
	static void SetUpPanoramicCharacter(AIdealTwinPanoramicCharacter* PanoramicChar, FImageObjectData ImageData, EImageRequestSize SizeImage, const FOnSphereSetUpCompleted& OnSphereSetUpCompletedDelegate);

	/**
	 * Create a new Panoramic Pawn and set up the image from image data
	 * @param WorldContext
	 * @param ViewMode 
	 * @param ImageData - Ideal Twin Image data to be downloaded
	 * @param SizeImage - Size used when requesting the image
	 * @param OnSphereSetUpCompletedDelegate 
	 * @param SpawnOffset - Set the offset position for the spawn
	 * @return return the new character create with the setup
	 */
	UFUNCTION(BlueprintCallable, Category= "IdealTwinCharater|IdealTwinPanoramicChar|Render", meta=(WorldContext="WorldContext"))
	static AIdealTwinPanoramicCharacter* CreateSetupPanoramicCharacter(UObject* WorldContext, TEnumAsByte<EITwPanoramicType> ViewMode, FImageObjectData ImageData, EImageRequestSize SizeImage, const FOnSphereSetUpCompleted& OnSphereSetUpCompletedDelegate, FVector
	                                                                   SpawnOffset = FVector(0, 0, -10000));
	
	UFUNCTION(BlueprintCallable, Category= "IdealTwinCharater|IdealTwinPanoramicChar")
	void SetBackgroundTexture(UTexture* BGTexture);

	UFUNCTION(BlueprintCallable, Category= "IdealTwinCharater|IdealTwinPanoramicChar")
	void ChangeViewType(TEnumAsByte<EITwPanoramicType> InViewType);
	
	virtual void PossessedBy(AController* NewController) override;
	virtual void UnPossessed() override;

#if WITH_EDITOR
	virtual void PostEditChangeProperty(FPropertyChangedEvent& PropertyChangedEvent) override;
	
#endif

protected:
	UFUNCTION()void Look360Character_Received(FVector2D LookValue);
	virtual void BeginPlay() override;


	UMaterial* GetMaterialByView();
	UFUNCTION() void OnDownloadedImage(FIdealTwinDownloadedImage DownloadedImage);
private:
	FOnSphereSetUpCompleted InternalSetUpDelegate;

	UPROPERTY()
	TObjectPtr<class AStaticMeshActor> DisplayMeshActor;
	UPROPERTY()
	TObjectPtr<UMaterialInstanceDynamic> DisplayDynInstance;
};
