// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "Kismet/KismetSystemLibrary.h"
#include "UObject/Object.h"
#include "IdealTwinViewConfig.generated.h"

class USpringArmComponent;

/** Encapsulates configuration options for managing camera positioning and angle adjustment while
 * ensuring proper visibility and collision detection. It is designed to prevent the camera from intersecting
 * with objects or the ground, and provides options for vision checks and height adjustments.
 */
USTRUCT(BlueprintType)
struct FAutoFixedAngleOptions
{
	GENERATED_BODY()

	/** Offset for calculating the camera vision position by creating an offset towards the camera position
	 * to prevent collisions with the ground or nearby objects */
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="AutoFixedAngleOptions")
	float SafeStartSpringLocationOffset;
	
	/** Radius of the SphereTrace used for checking vision between the spring arm and the camera */
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="AutoFixedAngleOptions")
	float VisionSphereRadius;
	
	/** Radius of the SphereTrace used for collision detection between a high point and the object */
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="AutoFixedAngleOptions")
	float FixedSphereRadius;
	
	/** Height added to checks that cast a trace from above downward.
	 * Applied to the initial position using e.g. StartLocation = GetCameraLocation() + (FVector::UpVector * Options.ExtraOffsetHeight); */
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="AutoFixedAngleOptions")
	float ExtraOffsetHeight;
	
	/** Offset applied to the final position if there are no obstacles in the line of sight and calculating from camera to ground.
	 * Adjusts the camera so it's not too close to the ground or an object. */
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="AutoFixedAngleOptions")
	FVector HitOffset;

	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="AutoFixedAngleOptions")
	TEnumAsByte<ETraceTypeQuery> TraceChannel;

	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="AutoFixedAngleOptions")
	bool bTraceComplex;
	
	/** If true, the angle will be applied to clamping. */
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="AutoFixedAngleOptions")
	bool bApplyAngle;

	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="AutoFixedAngleOptions")
	TEnumAsByte<EDrawDebugTrace::Type> Debug;

	UPROPERTY()
	TArray<AActor*> IgnoreActors;
	
	
	FAutoFixedAngleOptions(): SafeStartSpringLocationOffset(100), VisionSphereRadius(20), FixedSphereRadius(10),
	                          ExtraOffsetHeight(999999), HitOffset(FVector(0, 0, 100)), TraceChannel(ECC_Visibility),
	                          bTraceComplex(false),
	                          bApplyAngle(true), Debug(EDrawDebugTrace::None)
	{
	}
};


UENUM(BlueprintType,meta = (Bitflags, UseEnumValuesAsMaskValuesInEditor = "true"))
enum EViewConfigProperties
{
	EVCP_None					= 0				UMETA(Hidden),
	EVCP_UseZoom				= 1 << 0		UMETA(DisplayName="Use InitialZoom"),
	EVCP_UseClampViewAngle		= 1 << 1		UMETA(DisplayName="Use Clamp View Angle"),
	EVCP_UseSmoothCamera		= 1 << 2		UMETA(DisplayName="Use Smooth Camera"),
	EVCP_All					= EVCP_UseZoom |EVCP_UseClampViewAngle|EVCP_UseSmoothCamera	UMETA(Hidden),

};
ENUM_CLASS_FLAGS(EViewConfigProperties)

/**
 * Encapsulates settings for camera view angle clamping, including the initial rotation,
 * vertical and horizontal limits, and debugging options. These settings control the
 * constraints applied to camera movement to keep it within specific boundaries.
 */
USTRUCT(BlueprintType)
struct IDEALTWINPRO_API FClampSettings
{
	GENERATED_BODY()
	
	/** 
	 * Initial rotation of the camera view, used as a reference point for angle calculations
	 * and restoring the camera to its base state.
	 */
	UPROPERTY(BlueprintReadOnly, Category = "ClampView")
	FRotator StartViewRot;
	/** 
	 * Starting pitch angle for the camera. This defines the initial vertical orientation
	 * of the camera when the view configuration is first applied.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category = "ClampView", meta = (ClampMin="-85",ClampMax="85"))
	float StartAngle;
	/** 
	 * Maximum upward angle (negative pitch) the camera can rotate. Limits how far the
	 * player can look up, preventing the camera from going beyond this boundary.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category = "ClampView", DisplayName="UpClampAngle", meta = (ClampMin="-85",ClampMax="85"))
	float UpClampAngle;
	/** 
	 * Maximum downward angle (positive pitch) the camera can rotate. Limits how far the
	 * player can look down, preventing the camera from going beyond this boundary.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category = "ClampView", DisplayName="DownClampAngle", meta = (ClampMin="-85",ClampMax="85"))
	float DownClampAngle;
	/** 
	 * Maximum left angle (negative yaw) the camera can rotate. Limits how far the
	 * player can look to the left, restricting the horizontal rotation in that direction.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category = "ClampView", DisplayName="LeftClampAngle", meta = (ClampMin="0",ClampMax="179"))
	float LeftClampAngle;
	/** 
	 * Maximum right angle (positive yaw) the camera can rotate. Limits how far the
	 * player can look to the right, restricting the horizontal rotation in that direction.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category = "ClampView", DisplayName="RightClampAngle", meta = (ClampMin="0",ClampMax="179"))
	float RightClampAngle;

	
	/** Default constructor initializing all clamp angles to safe defaults */
	FClampSettings()
		: StartViewRot(FRotator::ZeroRotator)
		, StartAngle(35.f)
		, UpClampAngle(0.f)
		, DownClampAngle(0.f)
		, LeftClampAngle(0.f)
		, RightClampAngle(0.f)
	{
	}
};

/**
 * This class provides functionality for configuring and managing various view settings, such as zoom, camera clamping,
 * smoothing, and view configuration flags. Designed to work with actors or components that use a SpringArmComponent.
 */
UCLASS(Blueprintable,EditInlineNew)
class IDEALTWINPRO_API UIdealTwinViewConfig : public UObject
{
	GENERATED_BODY()

public:
	UIdealTwinViewConfig(const FObjectInitializer& ObjectInitializer);
	
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ViewConfig", meta = (Bitmask, BitmaskEnum =  "/Script/IdealTwinPro.EViewConfigProperties"))
	int32 ViewConfigProperties;
	
	///////////////////////////////////////////
	/// ZOOM UTILITIES
	///////////////////////////////////////////
	
	/// Use InitialZoom camera functions 

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ViewConfig|InitialZoom", DisplayName="Zoom",meta = (UIMin="0",ClampMin="0"))
	float InitialZoom;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ViewConfig|InitialZoom", meta = (UIMin="0",ClampMin="0"))
	float MinZoom;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ViewConfig|InitialZoom", meta = (UIMin="0",ClampMin="0"))
	float MaxZoom;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ViewConfig|InitialZoom")
	float StepZoom;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ViewConfig|InitialZoom")
	float InterpZoom;		
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ViewConfig|InitialZoom", meta = (AvancedDisplay))
	bool bInvertZoomDirection;
	
	///////////////////////////////////////////
	/// CLAMP VIEW UTILITIES
	///////////////////////////////////////////
	
	/** 
	 * Settings controlling the camera's rotational constraints, including vertical
	 * and horizontal boundaries and debugging options.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category = "ViewConfig|ClampView")
	FClampSettings ClampSettings;
	/** 
	 * When enabled, visual debugging aids are displayed to show the camera's clamp angles,
	 * helping developers visualize the boundaries of camera movement during development.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category = "ViewConfig|ClampView|Debug")
	bool bDebugClampAngles;

	

	/** 
	
	 * When enabled, the system automatically adjusts the downward camera clamp angle
	 * based on environmental conditions using trace checks. This helps prevent camera clipping
	 * through objects and ensures proper visibility by dynamically modifying the angle constraints
	 * based on detected obstacles or ground surfaces in the camera's path.
	 * 
	 * The behavior is configured through the AutoFixDownOptions property and is used
	 * by the AutoFixDownClampAngle function to perform the environmental checks.
	 * 
	 * Note: For this feature to work properly, the UpdateViewConfigTick() method must be called regularly
	 * (typically in a Tick function). AIdealTwinBaseCharacter handles this automatically in its Tick implementation.
	 * If you're using this class without AIdealTwinBaseCharacter, make sure to call UpdateViewConfigTick() in your
	 * own Tick function.
	 */

	/** Options for the AutoFixDownClampAngle function */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ViewConfig|ClampView")
	FAutoFixedAngleOptions AutoFixDownOptions;

	UFUNCTION(BlueprintCallable,Category="ViewConfig|ClampView")
	void SetUpClampAngle(float InUpClampAngle);
	UFUNCTION(BlueprintCallable,Category="ViewConfig|ClampView")
	void SetDownClampAngle(float InDownClampAngle);
	
	/** Sets the start view rotation in the clamp settings.
	 * @param InStartViewRot The rotation to set as the starting view rotation. */
	UFUNCTION(BlueprintCallable, Category="ViewConfig|ClampView")
	void SetStartViewRotation(FRotator InStartViewRot);
	FRotator SetStartViewRotation() const {return ClampSettings.StartViewRot;}
	
	UFUNCTION(BlueprintCallable,Category="ViewConfig|ClampView")
	float AutoFixDownClampAngle(FAutoFixedAngleOptions Options);
	
	
	/* If it's set true will be applied a smooth camera rotation and rotation*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite ,Category = "ViewConfig|SmoothCamera")
	bool bUseSmoothCamera;
	/* If it's set true apply a smooth camera movement*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category = "ViewConfig|SmoothCamera", meta=(InlineEditConditionToggle, EditCondition="bUseSmoothCamera"))
	bool bUseMovementSmooth;
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category = "ViewConfig|SmoothCamera", meta=(EditCondition="bUseMovementSmooth"))
	float LagMovement;
	/* If it's set true apply a smooth camera rotation*/
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category = "ViewConfig|SmoothCamera", meta=(InlineEditConditionToggle, EditCondition="bUseSmoothCamera"))
	bool bUseRotationSmooth;
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category = "ViewConfig|SmoothCamera", meta=(EditCondition="bUseRotationSmooth"))
	float LagRotation;


	/*	It scans if the owner is an actor or a component with a Spring arm to be set. If you create this object you must set
		the Outer at the constructor -> NewObject<UIdealTwinViewConfig>(this); */
	UFUNCTION()
	void ScanOwnerActor();
	/* Set the target SpringArm to be apply the view settings. */
	UFUNCTION(BlueprintCallable, Category="ViewConfig")
	void SetupSpringArm(USpringArmComponent* InSpringArm);


	UFUNCTION(BlueprintCallable, Category="ViewConfig")
	void BeginPlayViewConfig();

	
	/* Enable tick and bUsePawnControlRotation of the spring arm. */
	UFUNCTION(BlueprintCallable, Category = "ViewConfig")
	void SetActivateSpringArm(bool bActivate, bool bPawnControlRotation = true) const;

	/** Activates the view configuration settings, such as camera clamping, smoothing, adjusts the camera lag, rotation lag,
	 * and other properties based on the current view configuration flags.
	 * NOTE: Use ActivateViewConfigWithDelay between pawn possessions
	 * @param bEnableTick Indicates whether ticking should be enabled for the SpringArm. */
	UFUNCTION(BlueprintCallable, Category="ViewConfig")
	void ActivateViewConfig(bool bEnableTick = true);
	/** Activates the view configuration settings after a delay. This includes functionalities such as camera clamping,
	 * updating the controller rotation, and applying smooth tracking for movement and rotation.
	 * NOTE: Use this function instead of ActivateViewConfig between pawns possession to avoid crazy cameras movements.
	 *
	 * - Clears any existing timers related to smooth tracking or rotation updates.
	 * - Schedules a delayed timer for updating the controller's rotation and enabling specific view configurations.
	 * - Configures the SpringArmComponent for camera lag and rotation lag after a defined delay. */
	UFUNCTION(BlueprintCallable, Category="ViewConfig")
	void ActivateViewConfigWithDelay(AController* PawnController);
	/** Disables the current view configuration settings, such as camera clamping and SpringArm activation.
	 * This includes deactivating the SpringArmComponent and unclamping the camera rotation.
	 * Additionally, it stores the last controller rotation for potential future use.
	 */
	UFUNCTION(BlueprintCallable, Category="ViewConfig")
	void DeactivateViewConfig();

	/** Checks if a specific view configuration property is enabled in the current configuration.
	 * @param PropertyToCheck The property to check against the current view configuration settings.
	 *                        It should be one of the flags defined in EViewConfigProperties.
	 * @return True if the specified property is enabled in the current configuration, otherwise false. */
	UFUNCTION(BlueprintPure, Category = "ViewConfig")
	bool HasViewConfigProperty(EViewConfigProperties PropertyToCheck) const;	
	UFUNCTION(BlueprintCallable, Category = "ViewConfig")
	void RemoveViewConfigProperty(EViewConfigProperties PropertyToCheck);	
	UFUNCTION(BlueprintCallable, Category = "ViewConfig")
	void AddViewConfigProperty(EViewConfigProperties PropertyToCheck);


	/** Adjusts the current zoom level by applying the specified zoom delta.
	 * This function modifies the zoom settings based on the provided zoom delta,
	 * allowing for dynamic zoom changes in the view configuration.
	 *
	 * @param ZoomDelta The amount by which to adjust the zoom level. Positive values increase the zoom,
	 *                  while negative values decrease it. */
	UFUNCTION(BlueprintCallable, Category="ViewConfig")
	void ApplyZoomDelta(float ZoomDelta);

	/** Resets the current zoom level to the initial zoom value defined by the configuration.
	 * This method sets the internal zoom interpolation target to the specified initial zoom. */
	UFUNCTION(BlueprintCallable, Category="ViewConfig")
	void ResetZoomToInitial();
	/** Resets the rotation of the associated SpringArm to its initial value.
	 * This method sets the SpringArm's relative rotation to zero and restores the last known controller rotation
	 * to the initial view rotation. It is typically used to reset the camera orientation within the view configuration.*/
	UFUNCTION(BlueprintCallable, Category="ViewConfig")
	void ResetRotationToInitial();
	/** Resets the initial values for view configuration settings, including zoom and rotation, restoring them to their default state.*/
	UFUNCTION(BlueprintCallable, Category="ViewConfig")
	void ResetInitialValues();

	/** Updates the view configuration of the associated SpringArmComponent based on the current view configuration properties.
	 * Adjusts settings such as zoom, camera lag, and camera rotation lag if the corresponding configuration flags are set.*/
	UFUNCTION(BlueprintCallable, Category="ViewConfig")
	void UpdateViewConfig();
	/** This method updates the view configuration settings every tick, it modifies the TargetArmLength associated with
	 * the instance to achieve a smooth zoom effect.*/
	UFUNCTION(BlueprintCallable, Category="ViewConfig")
	void UpdateViewConfigTick();

	/** Adjusts the camera's rotational constraints by clamping or unclamping its pitch and yaw movements in APlayerCameraManager.
	 * @param bClamp A boolean value indicating whether to enable or disable clamping for the camera rotation.
	 *               If true, the camera's rotation is constrained to specified pitch and yaw ranges.
	 *               If false, the camera's rotation is unrestricted.
	 */
	UFUNCTION(BlueprintCallable, Category="ViewConfig")
	void ClampCameraRotation(bool bClamp);

	/** Sets the target zoom level for the view, clamping it within the allowable range defined by minimum and maximum zoom values.
	 * @param InZoomTarget The desired zoom level to set.*/
	UFUNCTION(BlueprintCallable, Category="ViewConfig")
	void SetZoomTarget(float InZoomTarget);

	/** Updates the rotation of the specified controller to match the last recorded control rotation.
	 * Intended to ensure consistent controller orientation within the view configuration context.
	 * @param Controller Pointer to the controller whose rotation will be updated. If null, the method does nothing. */
	void UpdateControllerRot(AController* Controller) const;
	
	// Timer functions for view configuration
	UFUNCTION()
	void UpdateRotationTimerFunction(AController* PawnController);
	UFUNCTION()
	void SmoothTrackTimerFunction() const;

	UFUNCTION(BlueprintCallable,Category="ViewConfig")
	USpringArmComponent* GetSpringArm() const;

	FClampSettings GetCurrentClampSettings() const { return CurrentClampSettings;}
	
	
#if WITH_EDITOR
	virtual void PostEditChangeProperty(FPropertyChangedEvent& PropertyChangedEvent) override;
	virtual void PostInitProperties() override;
	virtual void PostLoad() override;
	// Apply editor changes to the current SpringArm if exists
	virtual void EdSpringValues();
#endif
	

protected:
	void InitializeCameraRotation();
	AController* GetPawnController() const;

	
private:
	UPROPERTY() FRotator LastControllerRot;
	FClampSettings CurrentClampSettings;
	
	// Internal InitialZoom Target to be apply the lerp
	float ZoomTargetInterp;
	UPROPERTY() TSoftObjectPtr<USpringArmComponent> SpringArm;

	FTimerHandle Timer_UpdateRotation;
	FTimerHandle Timer_SmoothTrack;
	
};



// EXTRA CLASSES


UCLASS()
class IDEALTWINPRO_API UOrbitalViewConfig : public UIdealTwinViewConfig 
{
	GENERATED_BODY()

public:
	UOrbitalViewConfig(const FObjectInitializer& ObjectInitializer);
};
UCLASS()
class IDEALTWINPRO_API UWalkViewConfig : public UIdealTwinViewConfig 
{
	GENERATED_BODY()
public:
	UWalkViewConfig(const FObjectInitializer& ObjectInitializer);
};
UCLASS()
class IDEALTWINPRO_API UPanoramicViewConfig : public UIdealTwinViewConfig 
{
	GENERATED_BODY()
public:
	UPanoramicViewConfig(const FObjectInitializer& ObjectInitializer);
};
