﻿#pragma once

#include "CoreMinimal.h"
#include "GameFramework/PlayerController.h"
#include "IdealTwinBasePlayerController.generated.h"


class UIdealTwinInputComponent;
class USwitcherPawnComponent;
class UInputAction;
class UInputMappingContext;
DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnPawnChange, FString, NewPawnName);

UENUM(BlueprintType) enum EPlayerControllerInputMode
{
	EPCIM_GameAndUI		UMETA(DisplayName = "Game And UI", ToolTip="Set Game and UI input mode and set TRUE show mouse"),
	EPCIM_UIOnly		UMETA(DisplayName = "UI Only",ToolTip="Set UI Only input mode and set TRUE show mouse"),
	EPCIM_GameOnly		UMETA(DisplayName = "Game Only",ToolTip="Set Game Only input mode and set FALSE show mouse"),
	EPCIM_None			UMETA(DisplayName = "None",ToolTip="Don't set the input mode and keep default one")	
};

/** AIdealTwinBasePlayerController is a player controller base class that makes it easy to manage the
  	interactions with the actors since it manages an implementation with the ITwInteractionInterface interface and also
  	has record of InputMappingContext.
  	
  	Manage:
  	- UInputMappingContext registration
  	- Check Interaction Tick by timer
  	- Focus/Unfocus Actor With interface
  	- Can set input modes (Game and UI, UI Only, Game Only)		
 */
UCLASS()
class IDEALTWINPRO_API AIdealTwinBasePlayerController : public APlayerController
{
	GENERATED_BODY()

public:
	// Sets default values for this actor's properties
	AIdealTwinBasePlayerController();

	virtual void SetupInputComponent() override;
	
	/* Management component already integrated to make pawn switching by GameplayTags.*/
	UPROPERTY(VisibleInstanceOnly,BlueprintReadOnly,Category="IdealTwinPlayerController")
	TObjectPtr<USwitcherPawnComponent> SwitcherComponent;

	UPROPERTY(VisibleInstanceOnly, BlueprintReadOnly, Category="IdealTwinPlayerController|Input")
	TObjectPtr<UIdealTwinInputComponent> IdealTwinInputComponent;
	
	/* Set input management to begin play. To not use it set to NONE:
		- Game And UI : Set Game And UI Input Mode and Show mouse.
		- Game Only : Set Game Mode and NOT Show mouse.
		- UI Only : Set UI Input Mode and Show mouse.
		- NONE : Don't set the input mode and keep default one.
	 */
	UPROPERTY(BlueprintReadOnly, EditDefaultsOnly, Category="IdealTwinPlayerController")
	TEnumAsByte<EPlayerControllerInputMode> PlayerControllerInputMode;
	
	/*	When the PlayerController detect mouse is hovering a InteractionInterface with some actor it saves it. If there is
		no interaction with anything it will be null*/
	UPROPERTY(BlueprintReadOnly, VisibleDefaultsOnly, Category="IdealTwinPlayerController|Interaction")
	TObjectPtr<AActor> StoredInteractedActor;


	/*	If it is true, the PlayerController will register the UInputMappingContext and also use the inputs to do the management.
		of interactions with through the ITwInteracionInterface interface.
	 */
	UPROPERTY(BlueprintReadWrite, EditAnywhere, Category="IdealTwinPlayerController|Interaction")
	bool bUseInteractionSystem;
	/* Trace channel to check interaction*/
	UPROPERTY(BlueprintReadWrite, EditAnywhere, Category="IdealTwinPlayerController|Interaction")
	TEnumAsByte<ECollisionChannel> InteractionTraceChannel;
	


	/* Change the trace channel to check the interaction*/
	UFUNCTION(BlueprintCallable, Category="IdealTwinPlayerController|Interaction")
	void SetInteractionChannel(TEnumAsByte<ECollisionChannel> CollisionChannel);

	
	UFUNCTION() void TriggerPrimaryInteraction();
	UFUNCTION() void TriggerSecondaryInteraction();
	

	/* Execute the primary interaction with the StoredInteractedActor if it is not null by calling the interface ITwInteracionInterface*/
	UFUNCTION(BlueprintCallable, Category="IdealTwinPlayerController|Interaction")
	bool PrimaryInteraction();
	/* Execute the secondary interaction with the StoredInteractedActor if it is not null by calling the interface ITwInteracionInterface*/
	UFUNCTION(BlueprintCallable, Category="IdealTwinPlayerController|Interaction")
	bool SecondaryInteraction();	
	/*	Tick that executes the timer to check the interaction. If it is not overwritten in Blueprint check if the mouse
		is on top of an actor that implements ITwInteracionInterface and stores it in the StoredInteractedActor variable to be able to do
		the interaction with the click.*/
	UFUNCTION(BlueprintImplementableEvent, Category="IdealTwinPlayerController|Interaction")
	void InteractionTick(bool bResult, const FHitResult& HitResult);
	virtual void InteractionTick_Implementation(bool bResult, const FHitResult& HitResult);
	
	/* Gets the cursor hit information in the world to pass to the InteractionTick. */
	UFUNCTION() void CheckInteractionTick(FVector2D ScreenPosition);

	/* Handles the saving of the StoredInteractedActor.
		If the actor is valid, it saves it and calls the FocusActor event.
		If the actor is valid and there was already another previously focused perform a swap
		If the actor is NOT valid it will defocus the last actor saved and set it as null. */
	virtual void StoreInteractActor(AActor* InActor);
	/* Calls the interface event of IIdealTwinInteractionInterface::Execute_SetFocusActor*/
	void FocusInteractActor(bool bFocus) const;

	/* Toggle the interaction check or interaction timer turning it enable or disabled*/
	void ToggleCheckInteraction();

	UFUNCTION(BlueprintCallable,Category="IdealTwinPlayerController|Interaction")
	void SetControllerInputMode(TEnumAsByte<EPlayerControllerInputMode> NewInputMode);

	

protected:
	/* Registers the UInputMappingContext if the variable is true and also activates the timer if the timer check is used*/
	virtual void BeginPlay() override;

	virtual void OnPossess(APawn* InPawn);

	
	void InitializeControllerDesiredLocation();
	
private:

	
	/*Time between Press button and Release it to Detect if want move Camera o Click for interaction*/
	float TriggerSaveTime;
};




