// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "UObject/Interface.h"
#include "GameplayTags.h"
#include "SwitcherPawnComponent.h"
#include "Kismet/GameplayStatics.h"
#include "PawnSwitchInterface.generated.h"



// This class does not need to be modified.
UINTERFACE()
class UPawnSwitchInterface : public UInterface
{
	GENERATED_BODY()
};

/* This interface should be implemented in a Pawn class. It allows the Pawn to be controlled by the SwitcherPawnComponent. */
class IDEALTWINPRO_API IPawnSwitchInterface
{
	GENERATED_BODY()
public:

	/* Get a GameplayTag or Id from pawn to be register and be able possessed it by SwitcherComponent*/
	UFUNCTION(BlueprintNativeEvent, Category="PawnSwitchInterface")
	FGameplayTag GetSwitchPawnTag()const;

	/*Allows the auto-registration of the pawn class that is implementing this interface. DO NOT CALL THIS FUNCTION OUTSIDE THE CLASS*/
	UFUNCTION(BlueprintNativeEvent,BlueprintCallable, Category="PawnSwitchInterface",meta=(DefaultToSelf="OwnPawn", HidePin="OwnPawn"))
	bool AutoRegisterPawn(APawn* OwnPawn);
	virtual bool AutoRegisterPawn_Implementation(APawn* OwnPawn);

	
	/* This event is called when the pawn is about to be possessed, in case you want to execute some logic before completing the possession.*/
	UFUNCTION(BlueprintNativeEvent, Category="PawnSwitchInterface")
	void OnPrePossessSwitchPawn();
	/*This event is called when the pawn is about to be un-possessed, in case you want to execute some logic before completing the un-possession.*/
	UFUNCTION(BlueprintNativeEvent, Category="PawnSwitchInterface")
	void OnPreUnPossessSwitchPawn();

};

/**
 * Implements the auto-registration of a pawn with a SwitcherPawnComponent.
 *
 * This function attempts to automatically register the given Pawn (OwnPawn) with
 * the SwitcherPawnComponent associated with the PlayerController.
 *
 * @param OwnPawn The pawn instance that needs to be registered.
 * @return true if the Pawn was successfully registered; false otherwise.
 */
inline bool IPawnSwitchInterface::AutoRegisterPawn_Implementation(APawn* OwnPawn)
{
	APlayerController* PlayerController = UGameplayStatics::GetPlayerController(OwnPawn,0);
	if (PlayerController == nullptr) return false;

	USwitcherPawnComponent* SwitcherPawnComponent = Cast<USwitcherPawnComponent>(PlayerController->GetComponentByClass(USwitcherPawnComponent::StaticClass()));
	if (SwitcherPawnComponent == nullptr) return false;

	return SwitcherPawnComponent->RegisterPawnWithTag(Execute_GetSwitchPawnTag(OwnPawn) ,OwnPawn);
}
