// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "LevelVariantSets.h"
#include "Features/LevelVariant/IdealTwinConfiguratorDataAsset.h"
#include "UObject/Object.h"
#include "Widgets/Notifications/SNotificationList.h"
#include "IdealTwinEditorUtilitiesLibrary.generated.h"

UENUM()
enum EEditorNotificationCompletionState {
	CS_None,
	CS_Pending,
	CS_Success,
	CS_Fail,
}; 


/** Utilities use for the IdealTwin Editor workflow */
UCLASS()
class IDEALTWINPROEDITOR_API UIdealTwinEditorUtilitiesLibrary : public UBlueprintFunctionLibrary
{
	GENERATED_BODY()
public:
	// Configurator Utilities

	/**Creates a Configurator Data Asset from the specified Configurator ID.
	 * @param ConfiguratorID The ID of the configurator to generate a data asset for. */
	UFUNCTION(BlueprintCallable,Category="IdealTwinEditorUtilitiesLibrary",meta=(CallableWithoutWorldContext))
	static void CreateConfiguratorDataAssetFromConfigurator(const FName& ConfiguratorID);
	/** Creates or updates a LevelVariantSet using the given ConfiguratorData asset.
	 * This method generates a LevelVariantSet for the provided Configurator data asset.
	 * If the LevelVariantSet already exists, it updates it based on the Configurator information.
	 * If it does not exist, a new LevelVariantSet will be created.
	 * @param ConfiguratorData A reference to the UIdealTwinConfiguratorDataAsset used to create or update the LevelVariantSet.
	 *                         It must contain valid data, such as a ConfiguratorName or Room, to create the LevelVariantSet.
	 *                         If null or invalid, the operation will not proceed. */
	UFUNCTION(BlueprintCallable,Category="IdealTwinEditorUtilitiesLibrary",meta=(CallableWithoutWorldContext))
	static void CreateVariantSetFromConfigurator(UIdealTwinConfiguratorDataAsset* ConfiguratorData);
	/** Creates a Level Variant Actor in the current level using the specified configurator data.
	 * If a Level Variant Actor already exists in the level, it updates the existing actor with the provided configurator data.
	 * This function also manages viewport focus and actor selection in the editor.
	 * @param ConfiguratorData The configurator data asset used to set up the Level Variant Actor.
	 *                         If null, the function returns without performing any action. */
	UFUNCTION(BlueprintCallable,Category="IdealTwinEditorUtilitiesLibrary",meta=(CallableWithoutWorldContext))
	static void CreateLevelVariantActorInLevel(UIdealTwinConfiguratorDataAsset* ConfiguratorData);

	/** Updates the information of the given Configurator Data Asset.
	 * @param ConfiguratorData The IdealTwin Configurator Data Asset to update.
	 * @return True if the update was successful, false otherwise. */
	UFUNCTION(BlueprintCallable,Category="IdealTwinEditorUtilitiesLibrary",meta=(CallableWithoutWorldContext))
	static bool UpdateConfiguratorDataAssetInfo(UIdealTwinConfiguratorDataAsset* ConfiguratorData);

	/* Parse all data into the ConfiguratorDataAsset and apply those configurations to LevelVariantSets*/
	static bool FillOutLevelVariantWithConfigurator(UIdealTwinConfiguratorDataAsset* ConfiguratorData, ULevelVariantSets* LevelVariantSet);

	UFUNCTION(BlueprintCallable,Category="IdealTwinEditorUtilitiesLibrary",meta=(CallableWithoutWorldContext))
	static void PushEditorNotification(const FString& Title, const FString& SubMessage, TEnumAsByte<EEditorNotificationCompletionState> CompletionState, float ExpireDuration=7.f);
	//// Notification
	static void NotifyPopup(const FString& Title, const FString& SubMessage = "", SNotificationItem::ECompletionState NotifyType = SNotificationItem::CS_Success, float ExpireDuration=7.f);

	UFUNCTION(BlueprintCallable,Category="IdealTwinEditorUtilitiesLibrary",meta=(CallableWithoutWorldContext))
	static bool AssignRandomUnitsFromDataIntoLevel();

	UFUNCTION(BlueprintCallable,Category="IdealTwinEditorUtilitiesLibrary",meta=(CallableWithoutWorldContext))
	static void CopyToClipboard(const FString& TextToCopy, const FString& TextMessage);
	
	
	UFUNCTION(BlueprintCallable,Category="IdealTwinEditorUtilitiesLibrary",meta=(CallableWithoutWorldContext))
	static FString GetIdealTwinProVersion();
	UFUNCTION(BlueprintCallable,Category="IdealTwinEditorUtilitiesLibrary",meta=(CallableWithoutWorldContext))
	static FString GetIdealTwinSDKVersion();

	/** Verifies the existence of the "UnitVolume" collision profile and its associated configuration.
	 *
	 * This method checks if the necessary collision profile ("UnitVolume") and its default channel
	 * response are defined in the engine configuration files. Specifically, it ensures:
	 * - The "UnitVolume" response is added to the DefaultChannelResponses in the DefaultEngine.ini file.
	 * - The "UnitVolume" collision profile template exists within the collision profile system.
	 *
	 * If both configurations are present, a log entry is made to indicate success. Otherwise, an
	 * editor notification is displayed to alert the user and provide guidance for resolving the issue.
	 *
	 * Remarks:
	 * - This function does not make any modifications but communicates the readiness status of
	 *   the "UnitVolume" collision profile.
	 * - Missing configurations may require external intervention, such as updating configuration files or
	 *   regenerating profiles.
	 */
	UFUNCTION(BlueprintCallable,Category="IdealTwinEditorUtilitiesLibrary",meta=(CallableWithoutWorldContext))
	static void CheckForUnitVolumeCollisionProfile();
	/**
	 * Adds a new collision profile named "UnitVolume" with specific configurations.
	 *
	 * This function defines a collision profile template and ensures it is added
	 * to the collision profile system. Additionally, it updates the configuration
	 * file with the specified channel settings in the DefaultEngine.ini file.
	 * If the profile or configuration does not already exist, it modifies the file
	 * and prompts for a restart of the editor to apply the changes.
	 *
	 * The "UnitVolume" profile is configured specifically for query-only collision,
	 * where it blocks GameTraceChannel18 for click detection and ignores other collision settings.
	 *
	 * Remarks:
	 * - This operation might require a restart of the editor to take effect.
	 * - Notifications are triggered to indicate the success of the configuration
	 *   addition and to remind the user of the required restart.
	 */
	UFUNCTION(BlueprintCallable,Category="IdealTwinEditorUtilitiesLibrary",meta=(CallableWithoutWorldContext))
	static void AddUnitVolumeCollisionProfile();

	

	
};

/** Manages notifications related to collision settings validation and adjustment. */
UCLASS(Transient,NotBlueprintable,NotBlueprintType) class UCheckCollisionNotification : public UObject
{
	GENERATED_BODY()
public:
	void InitializeNotification();
	void ApplyCollisionSettings();
	void RestartEditor();
	void SkipCollisionSettings();
	TSharedPtr<SNotificationItem> NotificationItem;
};
