﻿// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "IdealTwinStyleModels.generated.h"


class UBorder;
class UTextBlock;

// FONT TYPES
USTRUCT(Blueprintable) struct IDEALTWINPROUI_API FIdealTwinFontSize
{
	GENERATED_BODY()
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinFontSize", meta=(ClampMin=1)) int32 FontSize;
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinFontSize") FName Typeface;
	FIdealTwinFontSize()
		: FontSize(21), Typeface("regular"){}
	FIdealTwinFontSize(int32 InFontSize, FName InTypeFace)
	: FontSize(InFontSize), Typeface(InTypeFace){}
};
USTRUCT(Blueprintable) struct IDEALTWINPROUI_API FIdealTwinFontStyle
{
	GENERATED_BODY()
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="IdealTwinFontSchema", meta=(AllowedClasses="/Script/Engine.Font"))
	const UObject* Font;
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="IdealTwinFontSchema")
	TMap<FName,FIdealTwinFontSize> FontSizing;

	FIdealTwinFontStyle(): Font(nullptr){}
};


// COLOR TYPE
UENUM(BlueprintType)enum EColorHandlerMode
{
	ECHM_ColorStyle				UMETA(DisplayName = "Color Style"),
	ECHM_ColorSimple			UMETA(DisplayName = "Color Simple"),
	ECHM_ColorCustom			UMETA(DisplayName = "Custom Color"),
	ECHM_Disable				UMETA(DisplayName = "Disable Color"),
};
USTRUCT(Blueprintable) struct IDEALTWINPROUI_API FIdealTwinColorHandler
{
	GENERATED_BODY()

	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="IdealTwinColorHandler")
	TEnumAsByte<EColorHandlerMode> ColorMode;
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="IdealTwinColorHandler")
	bool bUseContrast;
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="IdealTwinColorHandler",meta=(EditCondition="ColorMode!=EColorHandlerMode::ECHM_ColorCustom",EditConditionHides))
	FName ColorID;
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="IdealTwinColorHandler",meta=(EditCondition="ColorMode==EColorHandlerMode::ECHM_ColorCustom",EditConditionHides))
	FLinearColor ColorCustom;
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="IdealTwinColorHandler",meta=(InlineEditConditionToggle))
	bool bOverrideOpacity;
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="IdealTwinColorHandler", meta=(EditCondition="bOverrideOpacity"))
	float OpacityOverride;
	
	FLinearColor GetColor() const;
	FLinearColor GetContrastColor() const;

	FSlateColor GetColorSlate() const;
	FSlateColor GetContrastColorSlate() const;
	
	
	FIdealTwinColorHandler()
		: ColorMode(ECHM_ColorStyle),
		bUseContrast(false),
		ColorID(FName("Primary")),
		ColorCustom(FLinearColor::White),
		bOverrideOpacity(false),
		OpacityOverride(1)
	{}
	FIdealTwinColorHandler(const FName& InColor,bool bContrast = false)
	: ColorMode(ECHM_ColorStyle),
	bUseContrast(bContrast),
	ColorID(InColor),
	ColorCustom(FLinearColor::White),
	bOverrideOpacity(false),
	OpacityOverride(1)
	{}

	static FIdealTwinColorHandler TextColorHandler();
};

// FONT ICON TYPES
USTRUCT(BlueprintType) struct IDEALTWINPROUI_API FFontIconDatatableRow : public FTableRowBase
{
	GENERATED_BODY()
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="FontIconDatable") FString Label;
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="FontIconDatable") TArray<FName> Styles;
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="FontIconDatable") TArray<FName> Search;
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="FontIconDatable") FString Unicode;

	FFontIconDatatableRow(){}
};
USTRUCT(BlueprintType) struct IDEALTWINPROUI_API FFontIconData : public FTableRowBase
{
	GENERATED_BODY()

	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="FontIcon", meta=(AllowedClasses="/Script/Engine.Font"))
	const UObject* FontIcon;
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="FontIcon", meta=(RequiredAssetDataTags = "RowStructure=/Script/IdealTwinProUI.FontIconDatatableRow"))
	UDataTable* FontIconList;

	TArray<FName> GetFontTypeface() const;

	FFontIconData(): FontIcon(nullptr), FontIconList(nullptr)
	{}
};
USTRUCT(BlueprintType) struct IDEALTWINPROUI_API FFontIconHandler
{
	GENERATED_BODY()

	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="FontIconHandler") FName FontIconName;
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="FontIconHandler") FName IconStyle;
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="FontIconHandler") FName IconName;
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="FontIconHandler") FIdealTwinColorHandler IconColor;
	
	FFontIconHandler()
		: FontIconName("FontAwesomeFree")
		,IconStyle("Regular")
		,IconName("thumbs-up")
		,TableUnicode(nullptr)
	{}
	FFontIconHandler(FName InFontIconName ,FName InIconStyle,FName InIconName)
		: FontIconName(InFontIconName)
		,IconStyle(InIconStyle)
		,IconName(InIconName)
		,TableUnicode(nullptr)
	{}

	FSlateFontInfo& GenerateFontSlate(int IconSize = 45)const;
	FText GetIconUnicode()const;
	static FFontIconHandler GetDefaultConstruct();

private:
	UPROPERTY() mutable UDataTable* TableUnicode;
};


// SixeOverride
USTRUCT(Blueprintable) struct FIdealTwinSizeOverride
{
	GENERATED_BODY()
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="IdealTwinSizeOverride")
	bool bUseOverride;
	
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="IdealTwinSizeOverride", meta=(EditCondition="bUseOverride"))
	FVector2D SizeOverride;

	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="IdealTwinSizeOverride", meta=(InlineEditConditionToggle))
	bool bUseAspectRatio;
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="IdealTwinSizeOverride", meta=(EditCondition="bUseAspectRatio"))
	float AspectRatio;
	
	FIdealTwinSizeOverride()
		:bUseOverride(true), SizeOverride(FVector::ZeroVector), bUseAspectRatio(false),AspectRatio(1)
	{}

	void ApplyOverride(class USizeBox* InBox) const;
	void ApplyOverride(TSharedPtr<class SBox> InBox) const;

	
};

// BORDER
USTRUCT(Blueprintable) struct IDEALTWINPROUI_API FIdealTwinCornerStyle
{
	GENERATED_BODY()
	
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="IdealTwinCornerStyle")
	bool bUseUniformCorner;
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="IdealTwinCornerStyle", meta=(EditCondition="bUseUniformCorner", EditConditionHides))
	float UniformCorner;
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="IdealTwinCornerStyle", meta=(EditCondition="!bUseUniformCorner", EditConditionHides))
	FVector4 CornerRadius;
	UPROPERTY(EditAnywhere, BlueprintReadWrite,Category="IdealTwinCornerStyle")
	TEnumAsByte<enum ESlateBrushRoundingType::Type > RoundingType;

	FIdealTwinCornerStyle()
		:bUseUniformCorner(true),UniformCorner(5),CornerRadius(FVector4(5,5,5,5)), RoundingType(ESlateBrushRoundingType::FixedRadius)
	{}

	void SetCornerBrush(FSlateBrush& SlateBrush);
	void SetCornerBrushButton(FButtonStyle& SlateBrush);
};


// Components Styles
USTRUCT(Blueprintable) struct IDEALTWINPROUI_API FIdealTwinTextStyle
{
	GENERATED_BODY()
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinTextStyleSchema") FIdealTwinColorHandler ColorStyle;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinTextStyleSchema") FName FontSchema;
};
USTRUCT(Blueprintable) struct IDEALTWINPROUI_API FIdealTwinBorderStyle
{
	GENERATED_BODY()
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinBorderStyleSchema") FIdealTwinColorHandler ColorStyle;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinBorderStyleSchema") float Opacity;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinBorderStyleSchema") TEnumAsByte<enum ESlateBrushRoundingType::Type > RoundType;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinBorderStyleSchema") bool bUniformCorner = true;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinBorderStyleSchema") float UniformBorder;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinBorderStyleSchema") FVector4 CornerBorder;

	FIdealTwinBorderStyle()
		: ColorStyle("Background"),
		Opacity(1),
		RoundType(ESlateBrushRoundingType::FixedRadius),
		UniformBorder(5),
		CornerBorder(FVector4(5,5,5,5)){}
};

USTRUCT(Blueprintable) struct IDEALTWINPROUI_API FIdealTwinButtonStyle
{
	GENERATED_BODY()
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinButtonStyle") FIdealTwinColorHandler ColorStyle;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinButtonStyle") FIdealTwinCornerStyle CornerStyle;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinButtonStyle") FIdealTwinSizeOverride SizeOverride;
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="IdealTwinButtonStyle") float Opacity;

	FIdealTwinButtonStyle()
		: ColorStyle("Background"),
		Opacity(1)
	{}
};

UCLASS()
class IDEALTWINPROUI_API UIdealTwinStyleModels : public UBlueprintFunctionLibrary
{
	GENERATED_BODY()
public:
	UFUNCTION(BlueprintCallable, Category="IdealTwinStyleModels",  meta=(WorldContext="WorldContextObject", CallableWithoutWorldContext ))
	static void SetIdealTwinTextStyle(UTextBlock* Widget, FIdealTwinTextStyle Schema);
	
	UFUNCTION(BlueprintPure, Category="IdealTwinStyleModels",  meta=(WorldContext="WorldContextObject", CallableWithoutWorldContext ))
	static FSlateFontInfo GetFontInfoFromFontSchema(FName FontSchemaName);
	
	UFUNCTION(BlueprintCallable, Category="IdealTwinStyleModels",  meta=(WorldContext="WorldContextObject", CallableWithoutWorldContext ))
	static void SetIdealTwinBorderStyle(UBorder* Widget, FIdealTwinBorderStyle Schema);

	UFUNCTION(BlueprintCallable, Category="IdealTwinStyleModels",  meta=(WorldContext="WorldContextObject", CallableWithoutWorldContext ))
	static void SetIdealTwinButtonStyle(UButton* Button, FIdealTwinBorderStyle Schema);

};
