// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "Blueprint/UserWidget.h"
#include "Components/ListView.h"
#include "Features/LevelVariant/IdealTwinConfiguratorDataAsset.h"
#include "UObject/Object.h"
#include "ITwConfiguratorList.generated.h"


class AIdealTwinLevelVariantSetsActor;
class UIdealTwinVariantHandler;


UENUM(BlueprintType) enum EConfiguratorListElementType
{
	ECLET_Category			UMETA(DisplayName="Category"),
	ECLET_SubCategory		UMETA(DisplayName="SubCategory"),
	ECLET_Product			UMETA(DisplayName="Product"),
};

DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnVariantProductChange,int32,NewIndexVariant);
/* Payload para construir los items de la lista del configurador obteniendo el display text y el thumbnail.*/
UCLASS(NotBlueprintable, BlueprintType) class UITwConfiguratorListElement : public UObject
{
	GENERATED_BODY()
public:
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="ConfiguratorListElement") FIdealTwinItemOverrideData ItemData;
	UPROPERTY(EditAnywhere,BlueprintReadWrite,Category="ConfiguratorListElement") TEnumAsByte<EConfiguratorListElementType> ConfiguratorElementType;

	UPROPERTY(BlueprintAssignable,BlueprintReadWrite,Category="ConfiguratorListElement")
	FOnVariantProductChange OnVariantProductChange;

	
	static UITwConfiguratorListElement* CreateITwConfiguratorListElement();

	UFUNCTION(BlueprintCallable,Category="ConfiguratorListElement")
	void SetItemHasSelected();
	UFUNCTION(BlueprintPure,Category="ConfiguratorListElement")
	bool IsVariantSelected()const;

	UFUNCTION(BlueprintCallable,Category="ConfiguratorListElement")
	bool IsProductSelected();
	
	UFUNCTION(BlueprintCallable,Category="ConfiguratorListElement")
	UTexture2D* GetThumbnailFromVariant();
	UFUNCTION(BlueprintCallable,Category="ConfiguratorListElement")
	UObject* GetThumbnailFromOverride();
	UFUNCTION(BlueprintCallable, Category = "ConfiguratorListElement")
	bool GetProductDataFromVariant(FIdealTwinConfigProduct& OutProduct);


	UFUNCTION() void Received_VariantProductChange();

	/**
	 * Generates a list of product elements from a sub-category.
	 *
	 * @param SelectedElement Reference to the selected element.
	 * @return Array of product list elements derived from the provided sub-category.
	 */
	UFUNCTION(BlueprintCallable,Category="ConfiguratorListElement")
	TArray<UITwConfiguratorListElement*> GenerateProductListElementsFromSubCategory(UITwConfiguratorListElement*& SelectedElement);

	/**
	 * Checks for a selected item in the given array of configurator list elements.
	 *
	 * @param InItems Array of configurator list elements to check.
	 * @return Pointer to the selected configurator list element, or nullptr if no item is selected.
	 */
	UFUNCTION(BlueprintCallable,Category="ConfiguratorListElement")
	UITwConfiguratorListElement* CheckForSelectedItem(TArray<UITwConfiguratorListElement*>InItems);
	
	UPROPERTY(BlueprintReadOnly) TObjectPtr<UIdealTwinVariantHandler> VariantHandler;
	UPROPERTY() FName VariantActivated;
};



DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnConfiguratorElementSelected, UITwConfiguratorListElement*, ItemSelected);
/** Widget specialized to work closely with the AIdealTwinLevelVariantSetActor class to do the display of the 
	category and variant elements connected to it. */
UCLASS(Abstract)
class IDEALTWINPROUI_API UITwConfiguratorList : public UUserWidget
{
	GENERATED_BODY()
public:
	
	UPROPERTY(VisibleAnywhere,BlueprintReadOnly,Category="ConfiguratorList", meta=(BindWidget))
	TObjectPtr<UListView> ListDisplay;
	UPROPERTY(VisibleAnywhere,BlueprintReadOnly,Category="ConfiguratorList")
	TEnumAsByte<EConfiguratorListElementType> CurrentElementType;

	UPROPERTY(BlueprintAssignable)
	FOnConfiguratorElementSelected OnElementSelected;
	UFUNCTION(BlueprintImplementableEvent,Category="ConfiguratorList")
	void OnConfiguratorElementSelected(UITwConfiguratorListElement* ItemSelected);
	
	UFUNCTION(BlueprintCallable,Category="ConfiguratorList")
	void SetElementType(TEnumAsByte<EConfiguratorListElementType> ElementType);
	
	UFUNCTION(BlueprintCallable,Category="ConfiguratorList")
	void ResetToFirstLevel();

	void SetIdealTwinLevelVariantActor(AIdealTwinLevelVariantSetsActor* IdealTwinLevelVariant);

	
protected:
	virtual void NativeOnInitialized() override;
	virtual void NativeConstruct() override;

	void CheckForLevelVariantActor();
	void RefreshDisplayList() const;

	
	void ReceivedItemClick(UObject* Item);
	
	UPROPERTY(BlueprintReadOnly) TObjectPtr<AIdealTwinLevelVariantSetsActor> ITwLevelVariantSetRef;
};
