﻿// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "Components/Button.h"
#include "Style/IdealTwinStyleModels.h"
#include "ITwButton.generated.h"


class UITwFontIcon;
class UITwText;
class UScaleBox;


UENUM(BlueprintType) enum EButtonSelectionMode
{
	EBSM_None		UMETA(DisplayName = "None"),
	EBSM_Toggle		UMETA(DisplayName = "Toggle",ToolTip="Toggle the selection status of the button on every click"),
	EBSM_Blocked	UMETA(DisplayName = "Blocked",ToolTip="Keep blocked the button's selection until it's unselect via Blueprint"),
};

UENUM(BlueprintType, meta = (Bitflags, UseEnumValuesAsMaskValuesInEditor = "true"))
enum EButtonReactivityMode
{
	EBRM_None		= 0 UMETA(Hidden),
	EBRM_Toggle		= 1 << 0 UMETA(DisplayName = "Toggle"),
	EBRM_Hover		= 1 << 1 UMETA(DisplayName = "Hover"),
};
ENUM_CLASS_FLAGS(EButtonReactivityMode);
constexpr bool EnumHasAnyFlags(int32 Flags, EButtonReactivityMode Contains) { return (Flags & int32(Contains)) != 0; }
ENUM_RANGE_BY_VALUES(EButtonReactivityMode, 
	EButtonReactivityMode::EBRM_Toggle,
	EButtonReactivityMode::EBRM_Hover);

UCLASS()
class IDEALTWINPROUI_API UITwButton : public UButton
{
	GENERATED_BODY()
public:
	UITwButton(const FObjectInitializer& ObjectInitializer);

	DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnClickWithID, FName, ButtonID);
	UPROPERTY(BlueprintAssignable,Category="IdealTwinButton")
	FOnClickWithID OnClickWithID;
	DECLARE_DYNAMIC_MULTICAST_DELEGATE_TwoParams(FOnButtonSelectionChange, UITwButton*, ButtonChanged,bool,bSelected);
	UPROPERTY(BlueprintAssignable,Category="IdealTwinButton")
	FOnButtonSelectionChange OnButtonSelectionChange;

	UPROPERTY() int32 ButtonIndex;

	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinButton")
	bool bSelectedButton;
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinButton", meta=(ExposeOnSpawn))
	FName ButtonID;

	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinButton")
	TEnumAsByte<EButtonSelectionMode> SelectionMode;
	UFUNCTION(BlueprintCallable, Category="IdealTwinButton")
	void SetButtonSelectionMode(TEnumAsByte<EButtonSelectionMode> NewMode);
	UFUNCTION(BlueprintCallable, Category="IdealTwinButton")
	void SetSelectionState(bool bSelected, bool bNotificate = true);
	UFUNCTION(BlueprintCallable, Category="IdealTwinButton")
	bool IsButtonSelected()const;
	
	UPROPERTY(EditInstanceOnly, BlueprintReadWrite,Category="IdealTwinButton|Responsive", meta = (Bitmask, BitmaskEnum = EButtonReactivityMode))
	int32 ReactivityMode;
	
	UPROPERTY(EditInstanceOnly,BlueprintReadWrite, Category="IdealTwinButton|Responsive",meta=(BindWidgetOptional))
	TArray<TObjectPtr<UITwText>> ResponsiveText;
	UPROPERTY(EditInstanceOnly,BlueprintReadWrite, Category="IdealTwinButton|Responsive",meta=(BindWidgetOptional))
	TArray<TObjectPtr<UITwFontIcon>> ResponsiveIcons;
	
	
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinButton")
	FIdealTwinColorHandler Color;
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinButton")
	FIdealTwinCornerStyle CornerStyle;
	UPROPERTY(EditAnywhere,BlueprintReadWrite, Category="IdealTwinButton")
	FIdealTwinSizeOverride SizeOverride;

	UFUNCTION(BlueprintCallable, Category="IdealTwinButton")
	void SetButtonID(FName NewButtonID);
	
	UFUNCTION(BlueprintCallable, Category="IdealTwinButton")
	void SetButtonStyle(FIdealTwinButtonStyle InButtonStyle);
	
	
	UFUNCTION(BlueprintCallable, Category="IdealTwinButton")
	void UpdateStyle();


	void SetReactivity(bool bUseContrast);
	void SetReactivityHover(bool bUseContrast);

	
	virtual void ReleaseSlateResources(bool bReleaseChildren) override;
	virtual void SynchronizeProperties() override;
#if WITH_EDITOR
	virtual void PostEditChangeProperty(FPropertyChangedEvent& PropertyChangedEvent) override;
	virtual const FText GetPaletteCategory() override;
#endif

protected:
	
	// Needed to bind for keep the button selected
	void OnHoverReceived();
	void OnUnHoverReceived();
	
	FReply OnClickedReceived();
	
	virtual TSharedRef<SWidget> RebuildWidget() override;
private:
	TSharedPtr<SBox> BoxContainer;

};
