// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "Components/Widget.h"
#include "UObject/Object.h"
#include "ITwButtonGroup.generated.h"


class UITwButton;


UENUM(BlueprintType)
enum EButtonGroupControlMode
{
	EBGM_SingleSelection		UMETA(DisplayName = "Single Selection"),
	EBGM_MultiSelection			UMETA(DisplayName = "Multi Selection"),
	
};

DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnGroupClearSelection);
DECLARE_DYNAMIC_MULTICAST_DELEGATE_TwoParams(FOnGroupSelectionChange,bool,bEmptySelection, TArray<int32>, SelectionIndex);

class UITwButton;

/** Represents a UI component that manages a group of buttons, enabling behavior and selection control.
 * This class facilitates the creation of button groups with customizable selection modes, event bindings,
 * and selection management features. It supports both single and multi-selection capabilities, as well as
 * callback events for handling selection changes.
 *
 *NOTE: Ensure ITwButton are set as variable to add them as references into the array.
 */
UCLASS(HideCategories=(Behavior,RenderTransform,Rendering,Navigation,Localization,Performance))
class IDEALTWINPROUI_API UITwButtonGroup : public UWidget
{
	GENERATED_BODY()

public:
	UITwButtonGroup(const FObjectInitializer& ObjectInitializer);

	// Called when de selection have been cleaned. 
	UPROPERTY(BlueprintAssignable,Category="ITwButtonGroup")
	FOnGroupClearSelection OnClearSelection;
	// Called when the selection has changed.
	UPROPERTY(BlueprintAssignable,Category="ITwButtonGroup")
	FOnGroupSelectionChange OnSelectionChange;


	/** Represents the control mode for the button group within the ITwButtonGroup widget.
	 * This variable determines the selection behavior of the group, allowing either a single button
	 * to be selected at a time or multiple buttons to be selected simultaneously.
	 *
	 * Possible values:
	 * - EBGM_SingleSelection: Only one button can be selected at any given time.
	 * - EBGM_MultiSelection: Multiple buttons can be selected simultaneously.
	 *
	 * Modifying this variable will affect the behavior of the button group's selection logic.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="ITwButtonGroup")
	TEnumAsByte<EButtonGroupControlMode> ButtonGroupControlMode;

	/** A boolean flag that determines whether it is permissible to have no buttons selected within the group.
	 * If set to true, the button group can have zero selected buttons. If false, at least one button in the
	 * group must be selected at any given time. This setting is particularly relevant in both single and
	 * multi-selection modes for controlling the allowed selection behaviors.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="ITwButtonGroup")
	bool bAllowNoneSelection;
	/** An array of integers representing the indices of the buttons to be selected. For simple selection, only
	 * the first index is considered, whereas for multi-selection, multiple indices can be provided.*/
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="ITwButtonGroup")
	TArray<int32> SelectedButtonsIndex;

	// Override ButtonList and refresh references bindings
	UFUNCTION(BlueprintCallable, Category="ITwButtonGroup")
	void SetButtonList(TArray<UITwButton*> InButtonGroup);
	UFUNCTION(BlueprintCallable, Category="ITwButtonGroup")
	void AddButton(UITwButton* InButton);
	UFUNCTION(BlueprintCallable, Category="ITwButtonGroup")
	void RemoveButton(UITwButton* InButton);


	/**Retrieves a list of buttons that are currently associated with the button group.
	 * This method provides access to all buttons managed by the group, enabling
	 * further operations such as iteration or direct manipulation of the button instances.
	 *
	 * @return An array of UITwButton pointers representing the buttons within the group.
	 */
	UFUNCTION(BlueprintPure, Category="ITwButtonGroup")
	TArray<UITwButton*> GetButtonList()const;

	/** Determines whether the button group currently has any selected buttons.
	 *
	 * This method checks if the group contains one or more buttons in a selected state.
	 *
	 * @return true if there are selected buttons in the group, false otherwise.
	 */
	UFUNCTION(BlueprintPure, Category="ITwButtonGroup")
	bool HasSelection()const;
	/** Retrieves the IDs of the currently selected buttons within the button group.
	 *
	 * This method returns an array of identifiers corresponding to the buttons
	 * that are currently selected in the group. It ensures that only the IDs of buttons
	 * with active selections are included in the output.
	 *
	 * @return An array of FName representing the IDs of the selected buttons.
	 */
	UFUNCTION(BlueprintPure, Category="ITwButtonGroup")
	TArray<FName> GetSelectionButtonIDs() const;
	/** Retrieves the indices of the currently selected buttons in the group.
	 * If there are no selected buttons, the returned array will be empty.
	 * @return An array of integers representing the indices of the selected buttons in the button list.
	 */
	UFUNCTION(BlueprintPure, Category="ITwButtonGroup")
	TArray<int32> GetSelectionButtonIndexs() const;

	/**
	 * Sets the selected indices for the button group. This method updates the currently selected buttons based on the input array
	 * and clears any previous selections. It also triggers any required initialization for the updated selections.
	 *
	 * @param InSelection An array of integers representing the indices of the buttons to be selected. For simple selection, only
	 * the first index is considered, whereas for multi-selection, multiple indices can be provided.
	 */
	UFUNCTION(BlueprintCallable, Category="ITwButtonGroup")
	void SetSelectedIndex(TArray<int32> InSelection);

	
	UFUNCTION(BlueprintCallable, Category="ITwButtonGroup")
	void SetSelectionByButtonID(FName ButtonID);
	UFUNCTION(BlueprintCallable, Category="ITwButtonGroup")
	void SetSelectionByIndex(int32 ButtonIndex);
	UFUNCTION(BlueprintCallable, Category="ITwButtonGroup")
	void SetSelectionByButton(UITwButton* ButtonReference);

	UFUNCTION(BlueprintCallable, Category="ITwButtonGroup")
	void ClearAllButtonSelection();
	
	/* Clean all buttons in ButtonList by removing all event bindings and references */
	UFUNCTION(BlueprintCallable, Category="IdealTwin | Button Group")
	void CleanButtonList();

	/** Refreshes references to all buttons in the ButtonList.
	 * This method binds required delegates, assigns indices, and configures button selection modes based on the group control settings.
	 * It is also responsible for clearing and updating the SelectionButtons list while ensuring all references and bindings are valid.
	 */
	UFUNCTION(BlueprintCallable,Category="ITwButtonGroup")
	void RefreshButtonReferences();

	
#if WITH_EDITOR
	virtual void PostEditChangeChainProperty(struct FPropertyChangedChainEvent& PropertyChangedEvent) override;
	virtual const FText GetPaletteCategory() override;
#endif

protected:
	/** Holds a collection of buttons used within the button group for selection and interaction handling.
	 * This array manages references to UITwButton objects, enabling functionality such as selection tracking,
	 * event binding, and configuration of button group behavior within the group.
	 *
	 * NOTE: Populate this array with UITwButton objects to ensure proper grouping and interaction.
	 */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "ITwButtonGroup")
	TArray<UITwButton*> ButtonList;
	
	void CallOnSelectionChanged() const;

	UFUNCTION() void ReceivedOnSelectionChange(UITwButton* ButtonChanged, bool bSelected);

	virtual void OnSingleSelectionChange(UITwButton* ButtonChanged, bool bSelected);
	virtual void OnMultiSelectionChange(UITwButton* ButtonChanged, bool bSelected);

	virtual TSharedRef<SWidget> RebuildWidget() override;

	/** Initializes the selection state of buttons within the button group.
	 * This method iterates over the ButtonList and applies the appropriate selection state
	 * to each button based on the indices specified in SelectedButtonsIndex. Buttons with
	 * matching indices are marked as selected and added to the SelectionButtons map,
	 * while others are reset to the unselected state.
	 *
	 * Note:
	 * - This method ensures that the selected buttons' states and references are synchronized.
	 * - It is typically invoked during selection updates, button list refreshes, or certain
	 *   property changes in the button group.
	 */
	void InitialSelection();
private:
	// At some point, null objects are being added to the SelectionButtons array.This function identifies those null references and removes them from the array.
	void PruneInValidSelectionObject();
	
	UPROPERTY()
	TMap<int32,TObjectPtr<UITwButton>> SelectionButtons;
};


