// Fill out your copyright notice in the Description page of Project Settings.

#pragma once
#include "CoreMinimal.h"
#include "ITwMovableArea.h"
#include "Components/NamedSlot.h"
#include "UObject/Object.h"
#include "ITwZoomyArea.generated.h"

/* This Widget is an extension of SBox but with the particularity of being able to bind all mouse events for the
 * use of UWidget classes that need this information. */
class SBoxWithWheel : public SBox
{
public:
    SLATE_BEGIN_ARGS(SBoxWithWheel)
        : _AspectRatioUsesHeight(false)
        , _OnMouseWheel()
        , _OnMouseEnter()
        , _OnMouseLeave()
        , _OnMouseButtonDown()
        , _OnMouseButtonUp()
        , _WidthOverride(FOptionalSize())
        , _HeightOverride(FOptionalSize())
        , _MinDesiredWidth(FOptionalSize())
        , _MinDesiredHeight(FOptionalSize())
        , _MaxDesiredWidth(FOptionalSize())
        , _MaxDesiredHeight(FOptionalSize())
    {}
    	SLATE_ARGUMENT(bool, AspectRatioUsesHeight)
        SLATE_EVENT(FPointerEventHandler, OnMouseWheel)
        SLATE_EVENT(FSimpleDelegate, OnMouseEnter)
        SLATE_EVENT(FSimpleDelegate, OnMouseLeave)
        SLATE_EVENT(FPointerEventHandler, OnMouseButtonDown)
        SLATE_EVENT(FPointerEventHandler, OnMouseButtonUp)
    	SLATE_EVENT(FPointerEventHandler, OnMouseMove)
        SLATE_ATTRIBUTE(FOptionalSize, WidthOverride)
        SLATE_ATTRIBUTE(FOptionalSize, HeightOverride)
        SLATE_ATTRIBUTE(FOptionalSize, MinDesiredWidth)
        SLATE_ATTRIBUTE(FOptionalSize, MinDesiredHeight)
        SLATE_ATTRIBUTE(FOptionalSize, MaxDesiredWidth)
        SLATE_ATTRIBUTE(FOptionalSize, MaxDesiredHeight)
        SLATE_DEFAULT_SLOT(FArguments, Content)
    SLATE_END_ARGS()
    /** Constructs this widget with InArgs */
    void Construct(const FArguments& InArgs);
    /** Override events */
    virtual FReply OnMouseWheel(const FGeometry& MyGeometry, const FPointerEvent& MouseEvent) override;
	virtual FReply OnMouseMove(const FGeometry& MyGeometry, const FPointerEvent& MouseEvent) override;
    virtual void OnMouseEnter(const FGeometry& MyGeometry, const FPointerEvent& MouseEvent) override;
    virtual void OnMouseLeave(const FPointerEvent& MouseEvent) override;
    virtual FReply OnMouseButtonDown(const FGeometry& MyGeometry, const FPointerEvent& MouseEvent) override;
    virtual FReply OnMouseButtonUp(const FGeometry& MyGeometry, const FPointerEvent& MouseEvent) override;

private:
    FPointerEventHandler OnMouseWheelHandler;
	FPointerEventHandler OnMouseMoveHandler;
    FPointerEventHandler OnMouseButtonDownHandler;
    FPointerEventHandler OnMouseButtonUpHandler;
	FSimpleDelegate OnMouseEnterHandler;
	FSimpleDelegate OnMouseLeaveHandler;
};



DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnZoomValueChanged,float,ZoomValue);

/**
 * Represents a user interface component that provides zooming capabilities in a designated area.
 *
 * This class is designed to handle zoom interactions within a specified area of the user interface,
 * allowing for enhanced control and viewing of content. It can be used in applications where users
 * need to closely inspect or expand content within certain UI regions.
 *
 * Key features and functionalities of UITwZoomyArea include:
 * - Zoom in and zoom out capabilities.
 * - Adjustable zoom levels.
 * - Integration with other UI components for a seamless user experience.
 *
 * Note: Implementation specifics and additional customization options depend on the application
 * context and requirements.
 */
UCLASS()
class IDEALTWINPROUI_API UITwZoomyArea : public UContentWidget
{
	GENERATED_BODY()

public:
	UITwZoomyArea();


	/** Determines whether smooth transitions are enabled for zooming operations. */
	UPROPERTY(Editanywhere, BlueprintReadWrite, Category = "ITwZoomyArea")
	bool bEnableSmoothTransition;
	/** Specifies the speed of smooth transitions for zooming operations. It is only active when smooth transitions are enabled. */
	UPROPERTY(Editanywhere, BlueprintReadWrite, Category = "ITwZoomyArea", meta=(EditCondition = "bEnableSmoothTransition",EditConditionHides))
	float SmoothTransitionSpeed;

	/** Specifies the incremental step value for zooming operations within the ITwZoomyArea.
	 * This value determines how much the zoom level changes when zooming in or out. */
	UPROPERTY(Editanywhere, BlueprintReadWrite, Category = "ITwZoomyArea")
	float ZoomStep;


	/** Defines the permissible minimum and maximum zoom levels within the ITwZoomyArea.
	 * This vector establishes the range within which zoom operations are constrained,
	 * influencing both minimum reduction and maximum enlargement of the viewable area.*/
	UPROPERTY(Editanywhere, BlueprintReadWrite, Category = "ITwZoomyArea")
	FVector2D ZoomMinMax;
	
	UPROPERTY(BlueprintAssignable, Category="ITwZoomyArea")
	FOnZoomValueChanged OnZoomValueChanged;
	UPROPERTY(BlueprintAssignable, Category="ITwZoomyArea")
	FOnStartCaptureMouse OnStartCaptureMouse;
	UPROPERTY(BlueprintAssignable, Category="ITwZoomyArea")
	FOnStopCaptureMouse OnStopCaptureMouse;

	// Reset the content widget transformation to default values
	UFUNCTION(BlueprintCallable, Category="ITwZoomyArea")
	void ResetTransformation();
	// Reset the content widget translation to default values
	UFUNCTION(BlueprintCallable, Category="ITwZoomyArea")
	void ResetPosition();
	// Reset the content widget scale to default values
	UFUNCTION(BlueprintCallable, Category="ITwZoomyArea")
	void ResetScale();	

	UFUNCTION(BlueprintCallable, Category="ITwZoomyArea")
	void SetZoomValue(float NewZoom);
	UFUNCTION(BlueprintCallable, Category="ITwZoomyArea")
	float GetZoomValue()const;

	// Apply the action using the zoom step
	UFUNCTION(BlueprintCallable, Category="ITwZoomyArea")
	void ZoomOut();
	// Apply the action using the zoom step
	UFUNCTION(BlueprintCallable, Category="ITwZoomyArea")
	void ZoomIn();
	
	
	virtual void SynchronizeProperties() override;
	virtual void ReleaseSlateResources(bool bReleaseChildren) override;
#if WITH_EDITOR
	virtual const FText GetPaletteCategory() override;
	virtual void PostEditChangeProperty(FPropertyChangedEvent& PropertyChangedEvent) override;
#endif
	
protected:
	virtual TSharedRef<SWidget> RebuildWidget() override;

	FReply HandleMouseDown(const FGeometry& Geometry, const FPointerEvent& Event);
	FReply HandleMouseUp(const FGeometry& Geometry, const FPointerEvent& Event);
	FReply HandleMouseMove(const FGeometry& Geometry, const FPointerEvent& Event);
	FReply HandleMouseWheel(const FGeometry& Geometry, const FPointerEvent& Event);

	void CheckSmoothTimer();
	void SmoothTransitionTick();

	FVector2D GetMouseLocalPos(const FGeometry& Geometry) const;
private:
	
	bool bTrackMouse;
	FVector2D ClickOffset;
	FVector2D ContentStartTranslation;
	FTimerHandle SmoothTimerHandle;
	FWidgetTransform TargetTransform;
	
	TSharedPtr<SBoxWithWheel> MyWheelBox;
};


